#include "http.h"

#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/generic/reserve.h>
#include <util/generic/vector.h>
#include <util/string/ascii.h>

#include <utility>

namespace NSolomon {
namespace {
    // Not concurrent
    class TRequestHeaders final: public IHeaders {
        class THeader {
        public:
            THeader(size_t offset, size_t nameSize, size_t valueSize)
                : Offset_{offset}
                , NameSize_{nameSize}
                , ValueSize_{valueSize}
            {
            }

            TStringBuf Name(TStringBuf data) const {
                return data.SubStr(Offset_, NameSize_);
            }

            TStringBuf Value(TStringBuf data) const {
                return data.SubStr(Offset_ + NameSize_, ValueSize_);
            }

        private:
            size_t Offset_;
            size_t NameSize_;
            size_t ValueSize_;
        };

    public:
        TMaybe<TStringBuf> Find(TStringBuf name) const override {
            for (auto&& header: Headers_) {
                if (AsciiEqualsIgnoreCase(name, header.Name(Data_))) {
                    return header.Value(Data_);
                }
            }

            return Nothing();
        }

        void Add(TStringBuf name, TStringBuf value) override {
            Headers_.emplace_back(Data_.size(), name.size(), value.size());

            Data_ += name;
            Data_ += value;
        }

        void ForEach(std::function<void(TStringBuf, TStringBuf)> fn) const override {
            for (auto&& header: Headers_) {
                fn(header.Name(Data_), header.Value(Data_));
            }
        }

    private:
        // for a request it's unlikely that Find will be called heavily
        TVector<THeader> Headers_;
        TString Data_;
    };

    class TRequest final: public IRequest {
    public:
        TRequest(EHttpMethod method, TString url, TString data, IHeadersPtr headers)
            : Method_{method}
            , Url_{std::move(url)}
            , Data_{std::move(data)}
            , Headers_{headers ? std::move(headers) : IHeadersPtr(new TRequestHeaders)}
        {
        }

        static IRequestPtr Create(EHttpMethod method, TString url, TString data, IHeadersPtr headers) {
            return MakeHolder<TRequest>(method, std::move(url), std::move(data), std::move(headers));
        }

        EHttpMethod Method() const override {
            return Method_;
        }

        TStringBuf Url() const override {
            return Url_;
        }

        TStringBuf Data() const override {
            return Data_;
        }

        const IHeaders& Headers() const override {
            return *Headers_;
        }

        IHeaders& Headers() override {
            return *Headers_;
        }

    private:
        EHttpMethod Method_;
        TString Url_;
        TString Data_;
        IHeadersPtr Headers_;
    };

    template <typename TContainer>
    IHeadersPtr HeadersFromContainer(TContainer&& container) {
        auto h = MakeHolder<TRequestHeaders>();
        for (auto&& [name, value]: container) {
            h->Add(name, value);
        }

        return h;
    }

} // namespace

    IRequestPtr Get(TString url, IHeadersPtr headers) {
        return TRequest::Create(EHttpMethod::Get, std::move(url), {}, std::move(headers));
    }

    IRequestPtr Post(TString url, TString data, IHeadersPtr headers) {
        return TRequest::Create(EHttpMethod::Post, std::move(url), std::move(data), std::move(headers));
    }

    IRequestPtr Put(TString url, TString data, IHeadersPtr headers) {
        return TRequest::Create(EHttpMethod::Put, std::move(url), std::move(data), std::move(headers));
    }

    IRequestPtr CreateRequest(EHttpMethod method, TString url, TString data, IHeadersPtr headers) {
        return TRequest::Create(method, std::move(url), std::move(data), std::move(headers));
    }

    IHeadersPtr Headers(const TVector<std::pair<TString, TString>>& headers) {
        return HeadersFromContainer(headers);
    }

    IHeadersPtr Headers() {
        return MakeHolder<TRequestHeaders>();
    }
} // namespace NSolomon
