#pragma once

#include <solomon/libs/cpp/auth/actor/authentication_actor.h>
#include <solomon/libs/cpp/auth/core/internal_authorizer.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/http/http_proxy.h>

#include <utility>

namespace NSolomon::NHttp {

/**
 * Functor is accepting THttpIncomingRequestPtr and returning TString
 */
template <typename TFunc>
concept ReqHandleToStr = std::is_invocable_r_v<TString, TFunc, ::NHttp::THttpIncomingRequestPtr>;

/**
 * Functor is accepting THttpIncomingRequestPtr and returning THttpOutgoingResponsePtr
 */
template <typename TFunc>
concept ReqHandleToResp = std::is_invocable_r_v<::NHttp::THttpOutgoingResponsePtr, TFunc, ::NHttp::THttpIncomingRequestPtr>;

/**
 * Functor is accepting THttpIncomingRequestPtr and returning std::unique_ptr<NActors::IActor>
 */
template <typename TFunc>
concept ReqHandleFactory = std::is_invocable_r_v<std::unique_ptr<NActors::IActor>, TFunc, ::NHttp::THttpIncomingRequestPtr>;

/**
 * Base class for synchronous handlers, which can create immediate response.
 */
template <typename TFunc>
class TSyncHandler: public NActors::TActor<TSyncHandler<TFunc>> {
public:
    explicit TSyncHandler(TFunc&& func) noexcept
        : NActors::TActor<TSyncHandler>{&TSyncHandler<TFunc>::StateFunc}
        , Func_{std::forward<TFunc>(func)}
    {
    }

    STATEFN(StateFunc) {
        ui32 type = ev->GetTypeRewrite();
        Y_VERIFY(type == ::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::EventType,
                 "TSyncHandler cannot handle event %d", type);

        auto req = ev->CastAsLocal<::NHttp::TEvHttpProxy::TEvHttpIncomingRequest>()->Request;
        ::NHttp::THttpOutgoingResponsePtr resp;

        try {
            if constexpr (ReqHandleToResp<TFunc>) {
                resp = Func_(req);
            } else if constexpr (ReqHandleToStr<TFunc>) {
                TString content = Func_(req);
                resp = req->CreateResponseOK(content);
            } else {
                static_assert(TDependentFalse<TFunc>, "unsupported return type");
            }
        } catch (...) {
            resp = req->CreateResponse("500", "Internal server error", "text/plain", CurrentExceptionMessage());
        }

        this->Send(ev->Sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(std::move(resp)));
    }

private:
    TFunc Func_;
};

/**
 * Base class for asynchronous handlers, which need to create another actor to handle particular request.
 */
template <typename TFunc>
class TAsyncHandler: public NActors::TActor<TAsyncHandler<TFunc>> {
public:
    explicit TAsyncHandler(TFunc&& func) noexcept
        : NActors::TActor<TAsyncHandler>{&TAsyncHandler<TFunc>::StateFunc}
        , Func_{std::forward<TFunc>(func)}
    {
    }

    STFUNC(StateFunc) {
        ui32 type = ev->GetTypeRewrite();
        Y_VERIFY(type == ::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::EventType,
                 "TAsyncHandler cannot handle event %d", type);

        auto req = ev->CastAsLocal<::NHttp::TEvHttpProxy::TEvHttpIncomingRequest>()->Request;
        try {
            if (auto reqHandler = Func_(req)) {
                auto reqHandlerId = this->Register(reqHandler.release(), NActors::TMailboxType::Simple);
                ctx.Send(ev->Forward(reqHandlerId));
            } else {
                auto resp = req->CreateResponse("405", "Method Not Allowed", "text/plain");
                this->Send(ev->Sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(std::move(resp)));
            }
        } catch (...) {
            auto resp = req->CreateResponse("500", "Internal server error", "text/plain", CurrentExceptionMessage());
            this->Send(ev->Sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(std::move(resp)));
        }
    }

private:
    TFunc Func_;
};

/**
 * Base class for asynchronous handlers, which need to create another actor to handle particular request.
 */
template <typename THandler>
class TSimpleAsyncHandler: public NActors::TActor<TSimpleAsyncHandler<THandler>> {
public:
    TSimpleAsyncHandler()
        : NActors::TActor<TSimpleAsyncHandler>{&TSimpleAsyncHandler<THandler>::StateFunc}
    {
    }

    STFUNC(StateFunc) {
        ui32 type = ev->GetTypeRewrite();
        Y_VERIFY(type == ::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::EventType,
                 "TSimpleAsyncHandler cannot handle event %d", type);

        auto req = ev->CastAsLocal<::NHttp::TEvHttpProxy::TEvHttpIncomingRequest>()->Request;
        try {
            auto reqHandlerId = this->Register(new THandler{}, NActors::TMailboxType::Simple);
            ctx.Send(ev->Forward(reqHandlerId));
        } catch (...) {
            auto resp = req->CreateResponse("500", "Internal server error", "text/plain", CurrentExceptionMessage());
            this->Send(ev->Sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(std::move(resp)));
        }
    }
};

enum EAuthPolicy {
    NoAuth,
    AuthRequired
};

/**
 * Base class for handlers which are responsible to handle particular HTTP request.
 */
template <typename TDerived>
class TRequestHandler: public NActors::TActor<TDerived> {
protected:
    explicit TRequestHandler(
            TDuration timeout = TDuration::Seconds(10),
            EAuthPolicy authPolicy = EAuthPolicy::NoAuth,
            TActorId authenticationActor = {},
            NAuth::IInternalAuthorizerPtr internalAuthorizer = nullptr) noexcept
        : NActors::TActor<TDerived>{&TRequestHandler<TDerived>::Init}
        , Timeout_{timeout}
        , AuthPolicy_(authPolicy)
        , Auth_{authenticationActor}
        , InternalAuthorizer_(std::move(internalAuthorizer))
    {
    }

    STATEFN(Init) {
        switch (auto type = ev->GetTypeRewrite()) {
            hFunc(::NHttp::TEvHttpProxy::TEvHttpIncomingRequest, OnInit)
            default:
                Y_FAIL("%s cannot handle event %d", TypeName<TDerived>().c_str(), type);
        }
    }

    STFUNC(Work) {
        switch (ev->GetTypeRewrite()) {
            sFunc(NActors::TEvents::TEvWakeup, OnTimeout);
            hFunc(NSolomon::NAuth::TAuthEvents::TAuthenticationResponse, OnAuthenticationResponse)
            default:
                static_cast<TDerived*>(this)->Handle(ev, ctx);
        }
    }

    void OnInit(const ::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::TPtr& ev) {
        ReplyTo_ = ev->Sender;
        Request_ = ev->Get()->Request;
        this->Become(&TRequestHandler<TDerived>::Work);
        if (Timeout_) {
            this->Schedule(Timeout_, new NActors::TEvents::TEvWakeup);
        }
        if (AuthPolicy_ == EAuthPolicy::AuthRequired) {
            if (Auth_) {
                this->Send(Auth_, new NSolomon::NAuth::TAuthEvents::TAuthenticationRequest(Request_->Headers));
            } else {
                OnAuthenticationFailed();
            }
        } else {
            static_cast<TDerived*>(this)->AfterInit();
        }
    }

    void OnAuthenticationFailed() {
        ReplyAndDie(Request_->CreateResponse("401", "Unauthorized", "text/plain", "authentication failed"));
    }

    void OnAuthorizationFailed() {
        ReplyAndDie(Request_->CreateResponse("403", "Forbidden", "text/plain", "internal authorization failed"));
    }

    void OnAuthenticationResponse(const NSolomon::NAuth::TAuthEvents::TAuthenticationResponse::TPtr& ev) {
        if (ev->Get()->Success()) {
            if (InternalAuthorizer_) {
                const auto& authSubject = ev->Get()->GetAuthSubject();
                if (!InternalAuthorizer_->IsAllowed(authSubject)) {
                    OnAuthorizationFailed();
                    return;
                }
            }
            static_cast<TDerived*>(this)->AfterInit();
            return;
        }
        OnAuthenticationFailed();
    }

    void OnTimeout() {
        ReplyAndDie(Request_->CreateResponseGatewayTimeout());
    }

    void ReplyOk() {
        ReplyAndDie(Request_->CreateResponseOK("OK"));
    }

    void ReplyNotFound() {
        ReplyAndDie(Request_->CreateResponseNotFound());
    }

    void ReplyAndDie(::NHttp::THttpOutgoingResponsePtr resp) {
        this->Send(ReplyTo_, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse{std::move(resp)});
        this->PassAway();
    }

    ::NHttp::THttpIncomingRequest& Request() {
        return *Request_;
    }

    ::NHttp::THttpIncomingRequestPtr RequestPtr() {
        return Request_;
    }

private:
    TDuration Timeout_;
    TActorId ReplyTo_;
    const EAuthPolicy AuthPolicy_;
    const TActorId Auth_;
    ::NHttp::THttpIncomingRequestPtr Request_;
    NAuth::IInternalAuthorizerPtr InternalAuthorizer_; // may be null
};

template <typename THandler>
concept AsyncReqHandler = std::is_base_of_v<TRequestHandler<THandler>, THandler>;

} // namespace NSolomon::NHttp
