#include "http_server.h"

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>

#include <library/cpp/actors/http/http_proxy.h>

using namespace NActors;
using namespace NMonitoring;

using yandex::monitoring::HttpServerConfig;

namespace NSolomon::NHttp {

THttpServer::THttpServer(TActorRuntime& runtime, std::weak_ptr<TMetricRegistry> metrics, const HttpServerConfig& config)
    : ActorSystem_{runtime.ActorSystem()}
{
    Y_ENSURE(!config.executor_pool().empty(), "executor pool for the HTTP server must be specified");
    ExecutorId_ = runtime.FindExecutorByName(config.executor_pool());
    Proxy_ = ActorSystem_.Register(::NHttp::CreateHttpProxy(std::move(metrics)), TMailboxType::HTSwap, ExecutorId_);

    TString host = config.host();
    if (host.empty()) {
        host = "[::]";
    }

    Y_ENSURE(config.ports_size() > 0, "at least one port must be specified for http server");
    for (const auto& portConfig: config.ports()) {
        const ui32 port = portConfig.port();
        Y_ENSURE(port > 0 && port < Max<ui16>(), "port value " << port << " is out of range");
        auto message = std::make_unique<::NHttp::TEvHttpProxy::TEvAddListeningPort>(static_cast<ui16>(port), host);
        message->Secure = portConfig.secure();
        if (message->Secure) {
            message->CertificateFile = portConfig.cert_file();
            message->PrivateKeyFile = portConfig.key_file();
        }
        ActorSystem_.Send(Proxy_, message.release());
    }

    // TODO: check that port was taken
}

TActorId THttpServer::Handle(const TString& path, std::unique_ptr<IActor> handler) {
    auto handlerId = ActorSystem_.Register(handler.release(), TMailboxType::HTSwap, ExecutorId_, 0, Proxy_);
    ActorSystem_.Send(Proxy_, new ::NHttp::TEvHttpProxy::TEvRegisterHandler{path, handlerId});
    return handlerId;
}

TActorId THttpServer::Handle(const std::set<TString>& paths, std::unique_ptr<IActor> handler) {
    auto handlerId = ActorSystem_.Register(handler.release(), TMailboxType::HTSwap, ExecutorId_, 0, Proxy_);
    for (const auto& path: paths) {
        ActorSystem_.Send(Proxy_, new ::NHttp::TEvHttpProxy::TEvRegisterHandler{path, handlerId});
    }
    return handlerId;
}

} // namespace NSolomon::NHttp
