#include "metrics.h"

#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/http/server/core/response.h>
#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/auth/actor/authentication_actor.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/prometheus/prometheus.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/text/text.h>
#include <library/cpp/monlib/service/format.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NAuth;

namespace NSolomon::NHttp {
namespace {

/**
 * The main entrypoint to communicate with metrics handler service.
 */
const TActorId MetricsHandlerId{0, TStringBuf{"MetricsHdlr\0"}};

TErrorOr<IMetricEncoderPtr, TGenericError> CreateEncoder(EFormat fmt, ECompression compression, TStringStream* out) {
    switch (fmt) {
        case EFormat::JSON:
            return EncoderJson(out);

        case EFormat::SPACK:
            return EncoderSpackV1(out, ETimePrecision::SECONDS, compression);

        case EFormat::UNKNOWN:
        case EFormat::TEXT:
            return EncoderText(out);

        case EFormat::PROMETHEUS:
            return EncoderPrometheus(out);

        case EFormat::PROTOBUF:
            return TGenericError{"protobuf format is not supported"};
    }

    Y_UNREACHABLE();
}

class TReqWrapper {
public:
    explicit TReqWrapper(const ::NHttp::THttpIncomingRequest& req) noexcept
        : Params_{req.URL}
        , Headers_{req.Headers}
    {
    }

    const auto& GetParams() const {
        return Params_;
    }

    const auto& GetHeaders() const {
        return Headers_.Headers;
    }

private:
    ::NHttp::TUrlParameters Params_;
    ::NHttp::THeaders Headers_;
};

TString MakeFullResponse(EFormat fmt, ECompression compression, TStringBuf body) {
    return TStringBuilder{}
        << "HTTP/1.1 200 OK"
        << "\r\nConnection: Close"
        << "\r\nContent-Length: " << body.size()
        << "\r\nContent-Type: " << ContentTypeByFormat(fmt)
        << "\r\nContent-Encoding: " << ContentEncodingByCompression(compression)
        << "\r\n\r\n"
        << body;
}

struct TMetricHandlerEvents: public TPrivateEvents {
    enum {
        RegisterSupplier = SpaceBegin,
        End,
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TRegisterSupplier: public TEventLocal<TRegisterSupplier, RegisterSupplier> {
        std::weak_ptr<NMonitoring::IMetricSupplier> Metrics;

        explicit TRegisterSupplier(std::weak_ptr<NMonitoring::IMetricSupplier> metrics) noexcept
          : Metrics{std::move(metrics)}
        {
        }
    };
};

using TRequestId = ui64;

struct TInflightRequest {
    TActorId SenderId;
    ::NHttp::THttpIncomingRequestPtr Request;

    TInflightRequest() = default;

    TInflightRequest(TActorId sender, ::NHttp::THttpIncomingRequestPtr request)
        : SenderId(sender)
        , Request(std::move(request))
    {
    }
};

class TMetricHandlerActor: public TActorBootstrapped<TMetricHandlerActor> {
public:
    TMetricHandlerActor(
            std::shared_ptr<NMonitoring::IMetricSupplier> metrics,
            bool isAux,
            NActors::TActorId authenticationActor,
            IInternalAuthorizerPtr authorizer,
            bool authenticationOff)
        : MainMetrics_{std::move(metrics)}
        , AuthenticationActor_{authenticationActor}
        , Authorizer_{std::move(authorizer)}
        , IsAux_(isAux)
        , AuthenticationOff_{authenticationOff}
    {
    }

    void Bootstrap() {
        Become(&TThis::StateFunc);
        if (!IsAux_) {
            auto* actorSystem = TActorContext::ActorSystem();
            auto prevId = actorSystem->RegisterLocalService(MetricsHandlerId, SelfId());
            Y_VERIFY(!prevId, "multiple TMetricHandlerActor are registered");
        }
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TMetricHandlerEvents::TRegisterSupplier, OnRegisterSupplier)
            hFunc(::NHttp::TEvHttpProxy::TEvHttpIncomingRequest, OnHttpRequest)
            hFunc(TAuthEvents::TAuthenticationResponse, OnAuthenticationResponse)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRegisterSupplier(TMetricHandlerEvents::TRegisterSupplier::TPtr& ev) {
        auto& metrics = ev->Get()->Metrics;
        if (!metrics.expired()) {
            AuxMetrics_.push_back(metrics);
        }
    }

    void OnHttpRequest(::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::TPtr& ev) {
        MON_DEBUG(Http, "Start to process metrics request");
        auto& req = ev->Get()->Request;
        if (req->Method != TStringBuf{"GET"}) {
            Send(ev->Sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(NotAllowed(*req)));
            return;
        }

        if (AuthenticationActor_) {
            MON_DEBUG(Http, "Process metrics request with authentication");
            const ui64 cookie = PushRequest(ev);
            Send(AuthenticationActor_, new TAuthEvents::TAuthenticationRequest(req->Headers), 0, cookie);
        } else {
            MON_DEBUG(Http, "Process metrics request without authentication");
            ProcessRequest(ev->Sender, req.Get());
        }
    }

    void ProcessRequest(TActorId sender, ::NHttp::THttpIncomingRequest* req) {
        MON_DEBUG(Http, "Process metrics request");
        TReqWrapper reqWrapper{*req};
        auto fmt = ParseFormat(reqWrapper);
        auto compression = (fmt == NMonitoring::EFormat::SPACK)
                ? ParseCompression(reqWrapper)
                : ECompression::IDENTITY;

        TStringStream out;
        auto encoderOrErr = CreateEncoder(fmt, compression, &out);
        if (encoderOrErr.Fail()) {
            auto resp = req->CreateResponseBadRequest(encoderOrErr.Error().Message());
            Send(sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(std::move(resp)));
            return;
        }

        DumpMetrics(encoderOrErr.Extract());

        auto resp = req->CreateResponseString(MakeFullResponse(fmt, compression, out.Str()));
        Send(sender, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(std::move(resp)));
    }

    void LogFailedAuthorization(const TAuthSubject& authSubject) {
        TStringBuilder message;
        message << "Metrics request internal authorization failed. Type: " << authSubject.GetAuthType();
        switch (authSubject.GetAuthType()) {
            case NSolomon::NAuth::EAuthType::Unknown:
            case NSolomon::NAuth::EAuthType::OAuth:
                break;

            case NSolomon::NAuth::EAuthType::Iam:
                message << " Id: " << authSubject.AsIam().GetId();
                break;
            case NSolomon::NAuth::EAuthType::TvmService:
                if (authSubject.AsTvm().GetServiceTicket()->GetStatus() == NTvmAuth::ETicketStatus::Ok) {
                    message << " Id: " << authSubject.AsTvm().GetServiceTicket()->GetSrc();
                }
                break;
            case NSolomon::NAuth::EAuthType::TvmUser:
                break;
        }
        MON_DEBUG(Http, message);
    }

    void OnAuthenticationResponse(TAuthEvents::TAuthenticationResponse::TPtr& ev) {
        auto request = PopRequest(ev->Cookie);
        Y_VERIFY(request, "metrics handler got unknown authentication cookie %lu", ev->Cookie);

        if (ev->Get()->Success()) {
            MON_DEBUG(Http, "Metrics request authentication succeeded");
            const auto& authSubject = ev->Get()->GetAuthSubject();
            if (Authorizer_) {
                if (!Authorizer_->IsAllowed(authSubject)) {
                    LogFailedAuthorization(authSubject);
                    Send(request->SenderId, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(Forbidden(*request->Request)));
                    return;
                }
            }
            ProcessRequest(request->SenderId, request->Request.Get());
        } else {
            MON_DEBUG(Http, "Metrics request authentication failed");
            if (Y_UNLIKELY(AuthenticationOff_)) {
                ProcessRequest(request->SenderId, request->Request.Get());
            } else {
                Send(request->SenderId, new ::NHttp::TEvHttpProxy::TEvHttpOutgoingResponse(Unauthorized(*request->Request)));
            }
        }
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

    void DumpMetrics(IMetricEncoderPtr enc) {
        enc->OnStreamBegin();
        {
            if (MainMetrics_) {
                MainMetrics_->Append(TInstant::Zero(), enc.Get());
            }

            for (int i = static_cast<int>(AuxMetrics_.size()) - 1; i >= 0; i--) {
                if (auto metrics = AuxMetrics_[i].lock()) {
                    metrics->Append(TInstant::Zero(), enc.Get());
                } else {
                    // the reference is stale, remove it from the vector
                    AuxMetrics_[i] = std::move(AuxMetrics_.back());
                    AuxMetrics_.pop_back();
                }
            }
        }
        enc->OnStreamEnd();
    }

    TRequestId PushRequest(::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::TPtr& ev) {
        TRequestId id = NextRequestId_++;
        InflightRequests_[id] = TInflightRequest{ev->Sender, ev->Get()->Request};
        return id;
    }

    std::optional<TInflightRequest> PopRequest(TRequestId id) {
        auto it = InflightRequests_.find(id);
        if (it == InflightRequests_.end()) {
            return {};
        }
        auto req = std::move(it->second);
        InflightRequests_.erase(it);
        return req;
    }

private:
    std::shared_ptr<NMonitoring::IMetricSupplier> MainMetrics_;
    std::vector<std::weak_ptr<NMonitoring::IMetricSupplier>> AuxMetrics_;
    const TActorId AuthenticationActor_;
    std::unordered_map<TRequestId, TInflightRequest> InflightRequests_;
    TRequestId NextRequestId_ = 0;
    IInternalAuthorizerPtr Authorizer_;
    bool IsAux_;
    /**
     * This flag is a fallback. It must be false in normal situation.
     * May be true in configuration if authentication is damaged until it will recover.
     */
    const bool AuthenticationOff_ = false;
};

} // namespace

::NHttp::THttpOutgoingResponsePtr TMetricsHandler::operator()(const ::NHttp::THttpIncomingRequestPtr& req) {
    if (req->Method != TStringBuf{"GET"}) {
        return NotAllowed(*req);
    }

    TReqWrapper reqWrapper{*req};
    auto fmt = ParseFormat(reqWrapper);
    auto compression = (fmt == NMonitoring::EFormat::SPACK)
            ? ParseCompression(reqWrapper)
            : ECompression::IDENTITY;

    TStringStream out;
    auto encoderOrErr = CreateEncoder(fmt, compression, &out);
    if (encoderOrErr.Fail()) {
        return req->CreateResponseBadRequest(encoderOrErr.Error().Message());
    }

    {
        auto enc = encoderOrErr.Extract();
        Metrics_.Accept(TInstant::Zero(), enc.Get());
    }

    return req->CreateResponseString(MakeFullResponse(fmt, compression, out.Str()));
}

void RegisterMetricSupplier(const std::shared_ptr<NMonitoring::IMetricSupplier>& metrics) {
    auto* actorSystem = TActivationContext::ActorSystem();
    Y_VERIFY(actorSystem, "RegisterMetricSupplier() must be called inside the actor context");
    actorSystem->Send(MetricsHandlerId, new TMetricHandlerEvents::TRegisterSupplier{metrics});
}

std::unique_ptr<IActor> CreateMetricsHandler(
        std::shared_ptr<NMonitoring::IMetricSupplier> metrics,
        TMetricsAuthorizationConfig authorizationConfig)
{
    return std::make_unique<TMetricHandlerActor>(
            std::move(metrics),
            false, // isAux
            authorizationConfig.AuthenticationActor,
            std::move(authorizationConfig.Authorizer),
            authorizationConfig.AuthenticationOff);
}

std::unique_ptr<IActor> CreateAuxMetricsHandler(
        std::shared_ptr<NMonitoring::IMetricSupplier> metrics,
        TMetricsAuthorizationConfig authorizationConfig)
{
    return std::make_unique<TMetricHandlerActor>(
            std::move(metrics),
            true, // isAux
            authorizationConfig.AuthenticationActor,
            std::move(authorizationConfig.Authorizer),
            authorizationConfig.AuthenticationOff);
}

} // namespace NSolomon::NHttp
