#include "validator.h"

#include <contrib/libs/re2/re2/re2.h>

#include <util/string/builder.h>

namespace NSolomon {
namespace {
    constexpr size_t MAX_ID_LENGTH = 55;
    constexpr size_t MAX_SHARD_ID_LENGTH = 180;

    const auto VALID_ID_RE = re2::RE2{"[a-zA-Z0-9_-]+"};
    const auto NON_WHITESPACE_RE = re2::RE2{R"(\S+)"};
    const auto RESERVED_NAMES = re2::RE2{R"(new|protoNew)"};

    template <typename... TArgs>
    auto MakeError(TArgs&&... args) {
        return TGenericError{((TStringBuilder() << ... << std::forward<TArgs>(args)))};
    }

    class TIdValidator: public IIdValidator {
    public:
        TErrorOr<void, TGenericError> ValidateId(const TString& id) const override {
            return ValidateIdImpl(id, MAX_ID_LENGTH);
        }

        TErrorOr<void, TGenericError> ValidateShardId(const TString& shardId) const override {
            auto r = ValidateIdImpl(shardId, MAX_SHARD_ID_LENGTH);
            if (r.Fail()) {
                return r;
            }

            return {};
        }

    private:
        TErrorOr<void, TGenericError> ValidateIdImpl(const TString& id, size_t maxLength) const {
            if (id.empty() || !re2::RE2::PartialMatch(id, NON_WHITESPACE_RE)) {
                return MakeError("id cannot be blank");
            }

            if (re2::RE2::FullMatch(id, RESERVED_NAMES)) {
                return MakeError("id cannot be ", '"', id, '"');
            }

            if (id.size() > maxLength) {
                return MakeError("id must have length fewer than ", maxLength, " characters");
            }

            if (!re2::RE2::FullMatch(id, VALID_ID_RE)) {
                return MakeError("id must contain only latin letters, digits, hyphens and underscores");
            }

            return {};
        }

    };
} // namespace
    IIdValidatorPtr CreateIdValidator() {
        return MakeHolder<TIdValidator>();
    }
} // namespace NSolomon
