#include "arena.h"

#include <util/memory/mmapalloc.h>

namespace NSolomon::NIntern {

void* TArena::TPage::Alloc(size_t size, size_t align) {
    auto dataPtr = reinterpret_cast<uintptr_t>(Block.Data) + Offset;
    size_t padding = AlignUp(dataPtr, align) - dataPtr;

    if (size + padding > Block.Len - Offset) {
        return nullptr;
    }

    Offset += (size + padding);
    return reinterpret_cast<void*>(dataPtr + padding);
}

TArena::TArena(size_t pageSize, IAllocator* allocator /*= nullptr*/)
    : PageSize_(pageSize)
    , Allocator_(allocator ? allocator : MmapAllocator())
{
    Pages_.emplace_back(Allocator_->Allocate(pageSize));
}

TArena::~TArena() {
    for (TPage& page: Pages_) {
        Allocator_->Release(page.Block);
    }
}

void* TArena::Alloc(size_t size, size_t align) {
    if (auto* ptr = Pages_.back().Alloc(size, align)) {
        return ptr;
    }

    if (Y_LIKELY(size <= PageSize_)) {
        Pages_.emplace_back(Allocator_->Allocate(PageSize_));
        return Pages_.back().Alloc(size, align);
    }

    return nullptr;
}

void* TArena::PtrByOffset(size_t pageIdx, size_t pageOffset) const noexcept {
    if (Y_UNLIKELY(pageIdx >= Pages_.size())) {
        return nullptr;
    }

    const TPage& page = Pages_[pageIdx];
    if (Y_UNLIKELY(pageOffset > page.Offset)) {
        return nullptr;
    }

    auto dataPtr = reinterpret_cast<uintptr_t>(page.Block.Data) + pageOffset;
    return reinterpret_cast<void*>(dataPtr);
}

} // namespace NSolomon::NIntern
