#pragma once

#include <util/generic/noncopyable.h>
#include <util/generic/vector.h>
#include <util/system/align.h>

namespace NSolomon::NIntern {

class TArena: private TMoveOnly {
    struct TPage {
        IAllocator::TBlock Block;
        size_t Offset;

        explicit TPage(IAllocator::TBlock block) noexcept
            : Block(block)
            , Offset(0)
        {
        }

        void* Alloc(size_t size, size_t align);
    };

public:
    explicit TArena(size_t pageSize, IAllocator* allocator = nullptr);
    TArena(TArena&& oth) = default;
    TArena& operator=(TArena&& oth) = default;
    ~TArena();

    void* Alloc(size_t size) {
        return Alloc(size, PLATFORM_DATA_ALIGN);
    }

    void* Alloc(size_t size, size_t align);

    template <typename T>
    T* Put(T value) {
        T* ptr = static_cast<T*>(Alloc(sizeof(T), alignof(T)));
        *ptr = value;
        return ptr;
    }

    template <typename T>
    T Get(size_t index) const noexcept {
        size_t objectsPerPage = PageSize_ / sizeof(T);
        size_t pageIdx = index / objectsPerPage;
        size_t pageOffset = (index - pageIdx * objectsPerPage) * sizeof(T);
        return *reinterpret_cast<T*>(PtrByOffset(pageIdx, pageOffset));
    }

    template <typename T, typename = std::enable_if_t<std::is_pointer_v<T>>>
    T Ptr(size_t offset) const noexcept {
        size_t pageIdx = offset / PageSize_;
        size_t pageOffset = offset - pageIdx * PageSize_;
        return reinterpret_cast<T>(PtrByOffset(pageIdx, pageOffset));
    }

    size_t Offset() const noexcept {
        return PageSize_ * (Pages_.size() - 1) + Pages_.back().Offset;
    }

    template <typename T, typename... Args>
    T* New(Args&&... args) {
        T* ptr = static_cast<T*>(Alloc(sizeof(T), alignof(T)));
        new (ptr) T(std::forward<Args>(args)...);
        return ptr;
    }

    size_t AllocatedBytes() const noexcept {
        return Pages_.size() * PageSize_ +
               Pages_.capacity() * sizeof(TPage) +
               sizeof(*this);
    }

    size_t Pages() const noexcept {
        return Pages_.size();
    }

private:
    void* PtrByOffset(size_t pageIdx, size_t pageOffset) const noexcept;

private:
    const size_t PageSize_;
    IAllocator* const Allocator_;
    TVector<TPage> Pages_;
};

} // namespace NSolomon::NIntern
