#include <solomon/libs/cpp/intern/arena.h>
#include <solomon/libs/cpp/intern/rb_tree.h>

#include <library/cpp/testing/benchmark/bench.h>

#include <util/generic/map.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <util/random/random.h>

/*
Results from solomon-dev-myt-00.search.yandex.net
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

$ ya make -r && ./rb_tree --format csv | column -t

Name            Samples  Iterations  Run_time     Per_iteration_cycles  Per_iteration_sec
TMapInsert      376      87571       5.015936582  112778.0829           (113K)             cycles  -
TMapFind        10883    73126371    5.00142828   135.5140688           cycles             -
THashMapInsert  514      163878      5.003072546  60358.68232           (60.4K)            cycles  -
THashMapFind    18810    218436351   5.001986248  45.23325072           cycles             -
TRbMapInsert    271      45753       5.029634113  217703.4389           (218K)             cycles  -
TRbMapFind      9955     61189453    5.001220049  160.9620162           cycles             -

*/
struct TBenchTreeNode: public NSolomon::NIntern::TRbTreeNode<ui32, TBenchTreeNode> {
    ui32 Value;

    TBenchTreeNode(ui32 key, ui32 value)
        : NSolomon::NIntern::TRbTreeNode<ui32, TBenchTreeNode>(key)
        , Value(value)
    {
    }
};

struct TBenchTree: public NSolomon::NIntern::TRbTree<ui32, TBenchTreeNode> {
};

static const size_t KEYS = 1000;

static const struct TFixtures {
    NSolomon::NIntern::TArena Arena;
    TMap<ui32, ui32> Map;
    THashMap<ui32, ui32> HashMap;
    TBenchTree RbMap;
    TVector<ui32> RandomKeys;

    TFixtures()
        : Arena(4096)
    {
        for (ui32 key = 0; key < KEYS; key++) {
            ui32 value = key * 2;
            Map.emplace(key, value);
            HashMap.emplace(key, value);
            RbMap.Insert(Arena.New<TBenchTreeNode>(key, value));
            RandomKeys.push_back(RandomNumber<ui32>(KEYS));
        }
    }
} FIXTURES;

Y_CPU_BENCHMARK(TMapInsert, iface) {
    for (size_t i = 0; i < iface.Iterations(); i++) {
        TMap<ui32, ui32> map;
        for (ui32 key: FIXTURES.RandomKeys) {
            map.emplace(key, key * 2);
        }
        Y_DO_NOT_OPTIMIZE_AWAY(map);
    }
}

Y_CPU_BENCHMARK(TMapFind, iface) {
    auto& map = FIXTURES.Map;
    ui64 sum = 0;
    for (size_t i = 0; i < iface.Iterations(); i++) {
        auto it = map.find(RandomNumber<size_t>(KEYS));
        sum += (it != map.end() ? it->second : 0);
    }
    Y_DO_NOT_OPTIMIZE_AWAY(sum);
}

Y_CPU_BENCHMARK(THashMapInsert, iface) {
    for (size_t i = 0; i < iface.Iterations(); i++) {
        THashMap<ui32, ui32> map;
        for (ui32 key: FIXTURES.RandomKeys) {
            map.emplace(key, key * 2);
        }
        Y_DO_NOT_OPTIMIZE_AWAY(map);
    }
}

Y_CPU_BENCHMARK(THashMapFind, iface) {
    auto& map = FIXTURES.HashMap;
    ui64 sum = 0;
    for (size_t i = 0; i < iface.Iterations(); i++) {
        auto it = map.find(RandomNumber<size_t>(KEYS));
        sum += (it != map.end() ? it->second : 0);
    }
    Y_DO_NOT_OPTIMIZE_AWAY(sum);
}

Y_CPU_BENCHMARK(TRbMapInsert, iface) {
    for (size_t i = 0; i < iface.Iterations(); i++) {
        TBenchTree map;
        NSolomon::NIntern::TArena arena(4096);
        for (ui32 key: FIXTURES.RandomKeys) {
            map.Insert(arena.New<TBenchTreeNode>(key, key * 2));
        }
        Y_DO_NOT_OPTIMIZE_AWAY(map);
    }
}

Y_CPU_BENCHMARK(TRbMapFind, iface) {
    auto& map = FIXTURES.RbMap;
    ui64 sum = 0;
    for (size_t i = 0; i < iface.Iterations(); i++) {
        if (auto* node = map.Find(RandomNumber<size_t>(KEYS))) {
            sum += node->Value;
        }
    }
    Y_DO_NOT_OPTIMIZE_AWAY(sum);
}
