#include <solomon/libs/cpp/intern/str_pool.h>

#include <library/cpp/testing/benchmark/bench.h>

#include <util/generic/vector.h>
#include <util/generic/hash_set.h>
#include <util/random/random.h>
#include <util/random/fast.h>

/*
Results from solomon-dev-myt-00.search.yandex.net
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

 $ ya make -r && ./str_pool --format csv | column -t

 Name                 Samples  Iterations  Run_time     Per_iteration_cycles  Per_iteration_sec
 Intern_EmptyPool     5157     16419315    5.000616993  644.6422296           cycles             -
 Intern_NonEmptyPool  7261     32550346    5.001388378  318.4099353           cycles             -
 FindById             15403    146470170   5.002118491  66.7404051            cycles             -
 FindByStr            7628     35925526    5.001195312  286.2778544           cycles             -
*/

using namespace NSolomon::NIntern;

static const ui32 SIZE = 10000;

static struct TFixtures {
    TVector<TString> UniqueStrings;
    TStringPool PoolA, PoolB;
    TVector<TStringId> Ids;
    TVector<TString> MoreStrings;

    TFixtures() {
        THashSet<TString> strings;
        for (ui32 seed = 0; strings.size() < SIZE; seed++) {
            for (ui32 len: std::initializer_list<ui32>{3, 5, 7, 11}) {
                strings.insert(RandomString(len, seed));
            }
        }

        UniqueStrings.reserve(SIZE);
        Ids.reserve(SIZE);

        for (const TString& s: strings) {
            UniqueStrings.push_back(s);
            Ids.push_back(PoolA.Intern(s));
            PoolB.Intern(s);
        }

        MoreStrings.reserve(2 * SIZE);

        for (ui32 seed = Max<ui32>() - SIZE; MoreStrings.size() < 2 * SIZE; seed++) {
            for (ui32 len: std::initializer_list<ui32>{3, 5, 7, 11}) {
                MoreStrings.push_back(RandomString(len, seed));
            }
        }
    }

    static TString RandomString(size_t len, ui32 seed) {
        TReallyFastRng32 rand(seed);
        TString ret(len, '\0');
        for (size_t i = 0; i < len; ++i) {
            ret[i] = static_cast<char>(rand.Uniform(1, 128));
        }
        return ret;
    }

} FIXTURES;

Y_CPU_BENCHMARK(Intern_EmptyPool, iface) {
    TStringPool pool;
    for (ui32 i = 0; i < iface.Iterations(); i++) {
        pool.Intern(FIXTURES.UniqueStrings[i % SIZE]);
    }
    Y_DO_NOT_OPTIMIZE_AWAY(pool);
}

Y_CPU_BENCHMARK(Intern_NonEmptyPool, iface) {
    TStringPool* pool = &FIXTURES.PoolA;
    for (ui32 i = 0; i < iface.Iterations(); i++) {
        pool->Intern(FIXTURES.MoreStrings[i % SIZE]);
    }
    Y_DO_NOT_OPTIMIZE_AWAY(pool);
}

Y_CPU_BENCHMARK(FindById, iface) {
    TStringPool* pool = &FIXTURES.PoolB;
    ui64 sink = 0;
    for (ui32 i = 0; i < iface.Iterations(); i++) {
        auto str = pool->Find(RandomNumber<TStringId>(SIZE));
        sink += str.size();
    }
    Y_DO_NOT_OPTIMIZE_AWAY(sink);
}

Y_CPU_BENCHMARK(FindByStr, iface) {
    TStringPool* pool = &FIXTURES.PoolB;
    ui64 sink = 0;
    for (ui32 i = 0; i < iface.Iterations(); i++) {
        auto id = pool->Find(FIXTURES.UniqueStrings[i % SIZE]);
        sink += id;
    }
    Y_DO_NOT_OPTIMIZE_AWAY(sink);
}
