#include "str_pool.h"
#include "arena.h"
#include "rb_tree.h"
#include "str_ref.h"

#include <util/digest/city.h>
#include <util/generic/strbuf.h>

#include <cstring>

namespace NSolomon::NIntern {
namespace {

static constexpr size_t PageSize = 4096;

inline ui64 StringHash(TStringBuf str) {
    return CityHash64(str);
}

// XXX: lost 4 bytes due to padding
struct TIndexItem {
    const TStringId Id;
    const TStringRef Str;
    TIndexItem* Next; // hash collision linked list

    TIndexItem(TStringId id, TStringRef str)
        : Id(id)
        , Str(str)
        , Next(nullptr)
    {
    }
};
static_assert(sizeof(TIndexItem) == 24, "sizeof(TIndexItem) != 24");

/**
 * Index node combining RB-tree node to search by hash value
 * and single linked list to handle hash collisions (actually preaty rare
 * with CityHash function).
 */
struct TIndexNode: public TRbTreeNode<ui64, TIndexNode> {
    using TBase = TRbTreeNode<ui64, TIndexNode>;

    TIndexItem Item;

    TIndexNode(TStringId id, TStringRef str, ui64 hash)
        : TBase(hash)
        , Item(id, str)
    {
    }
};
static_assert(sizeof(TIndexNode) == 48, "sizeof(TIndexNode) != 48");

using TIndexTree = TRbTree<ui64, TIndexNode>;

} // namespace


class TBaseImpl {
public:
    TBaseImpl()
        : Offsets_(PageSize, TDefaultAllocator::Instance())
        , Strings_(PageSize, TDefaultAllocator::Instance())
        , Index_(PageSize, TDefaultAllocator::Instance())
        , Size_(0)
    {
    }

    TStringId Intern(TStringBuf str) {
        ui64 hash = StringHash(str);
        if (TIndexNode* node = IndexTree_.Find(hash)) {
            if (node->Item.Str == str) {
                return node->Item.Id;
            }

            // handle hash collision with different strings
            TIndexItem* item = &node->Item;
            while (item->Next) {
                item = item->Next;
                if (item->Str == str) {
                    return item->Id;
                }
            }

            Y_ASSERT(!item->Next);
            auto* newItem = StoreString<TIndexItem>(hash, str);
            item->Next = newItem;
            return newItem->Id;
        }

        auto* node = StoreString<TIndexNode>(hash, str);
        IndexTree_.Insert(node);
        return node->Item.Id;
    }

    TStringBuf Find(TStringId id) const noexcept {
        if (Y_UNLIKELY(id == InvalidStringId || id > static_cast<TStringId>(Size_))) {
            return {};
        }

        ui64 offset = Offsets_.Get<ui64>(static_cast<size_t>(id) - 1);
        return TStringRef::CreateFromOffset(Strings_, offset);
    }

    TStringId Find(TStringBuf str) const noexcept {
        ui64 hash = StringHash(str);
        if (TIndexNode* node = IndexTree_.Find(hash)) {
            TIndexItem* item = &node->Item;
            do {
                if (item->Str == str) {
                    return item->Id;
                }
            } while ((item = item->Next) != nullptr);
        }
        return InvalidStringId;
    }

    size_t Size() const noexcept {
        return Size_;
    }

    size_t AllocatedBytes() const noexcept {
        return Offsets_.AllocatedBytes() +
               Strings_.AllocatedBytes() +
               Index_.AllocatedBytes();
    }

protected:
    template <typename T>
    T* StoreString(ui64 hash, TStringBuf str) {
        TStringId id = static_cast<TStringId>(Size_) + 1;
        Y_ENSURE(id != InvalidStringId, "more than " << Max<TStringId>() << " unique strings were interned");

        auto [strCopy, offset] = TStringRef::CreateFromString(Strings_, str);
        Offsets_.Put(offset);

        T* result;
        if constexpr (std::is_same_v<T, TIndexNode>) {
            result = Index_.New<T>(id, strCopy, hash);
        } else {
            result = Index_.New<T>(id, strCopy);
        }

        Y_ENSURE(result, "cannot allocate index node/item");
        ++Size_;
        return result;
    }

    TArena Offsets_;
    TArena Strings_;
    TArena Index_;
    TIndexTree IndexTree_;
    size_t Size_;
};

template <>
struct TStringPoolBase<void>::TImpl: public TBaseImpl {
public:
    using TBase = TBaseImpl;

    size_t AllocatedBytes() const noexcept {
        return TBase::AllocatedBytes() + sizeof(*this);
    }
};

template <>
struct TStringPoolBase<TLightRWLock>::TImpl: public TBaseImpl {
    using TBase = TBaseImpl;

    TStringId Intern(TStringBuf str) {
        {
            TLightReadGuard g(Lock_);
            if (TStringId id = TBase::Find(str); id != InvalidStringId) {
                return id;
            }
        }

        TLightWriteGuard g(Lock_);
        return TBase::Intern(str);
    }

    TStringBuf Find(TStringId id) const noexcept {
        TLightReadGuard g(Lock_);
        return TBase::Find(id);
    }

    TStringId Find(TStringBuf str) const noexcept {
        TLightReadGuard g(Lock_);
        return TBase::Find(str);
    }

    size_t Size() const noexcept {
        TLightReadGuard g(Lock_);
        return TBase::Size();
    }

    size_t AllocatedBytes() const noexcept {
        TLightReadGuard g(Lock_);
        return TBase::AllocatedBytes() + sizeof(*this);
    }

private:
    TLightRWLock Lock_;
};

struct TRCUStringPool::TImpl {
public:
    TStringId Intern(TStringBuf str) {
        TStringId id = ReadOnly_.Find(str);
        if (id != InvalidStringId) {
            return id;
        }

        {
            TLightReadGuard g(Lock_);
            id = Mutable_.Find(str);

            if (id != InvalidStringId) {
                return id;
            }
        }

        TLightWriteGuard g(Lock_);
        id =  Mutable_.Intern(str);
        Updates_.emplace_back(id);

        return id;
    }

    TStringId Find(TStringBuf str) const noexcept {
        TStringId id = ReadOnly_.Find(str);
        if (id != InvalidStringId) {
            return id;
        }

        TLightReadGuard g(Lock_);
        return Mutable_.Find(str);
    }

    TStringBuf Find(TStringId id) const noexcept {
        TStringBuf str = ReadOnly_.Find(id);
        if (str) {
            return str;
        }

        TLightReadGuard g(Lock_);
        return Mutable_.Find(id);
    }

    size_t Size() const noexcept {
        TLightReadGuard g(Lock_);
        return Mutable_.Size();
    }

    size_t AllocatedBytes() const noexcept {
        TLightReadGuard g(Lock_);
        return ReadOnly_.AllocatedBytes()
                + Mutable_.AllocatedBytes()
                + Updates_.capacity() * sizeof(TStringId)
                + sizeof(*this);
    }

    void Update() {
        for (auto id: Updates_) {
            TStringId r = ReadOnly_.Intern(Mutable_.Find(id));
            Y_VERIFY(r == id);
        }

        Updates_.clear();
    }

private:
    static constexpr ui32 CACHE_LINE_BYTES = 64;

private:
    TBaseImpl ReadOnly_;
    TBaseImpl Mutable_;
    std::vector<TStringId> Updates_;

    Y_DECLARE_UNUSED char __[CACHE_LINE_BYTES];
    TLightRWLock Lock_;
};

template <typename TLock>
TStringPoolBase<TLock>::TStringPoolBase()
    : Impl_(new TImpl)
{
}

template <typename TLock>
TStringPoolBase<TLock>::~TStringPoolBase() {
}

template <typename TLock>
TStringId TStringPoolBase<TLock>::Intern(TStringBuf str) {
    return Impl_->Intern(str);
}

template <typename TLock>
TStringBuf TStringPoolBase<TLock>::Find(TStringId id) const noexcept {
    return Impl_->Find(id);
}

template <typename TLock>
TStringId TStringPoolBase<TLock>::Find(TStringBuf str) const noexcept {
    return Impl_->Find(str);
}

template <typename TLock>
size_t TStringPoolBase<TLock>::Size() const noexcept {
    return Impl_->Size();
}

template <typename TLock>
size_t TStringPoolBase<TLock>::AllocatedBytes() const noexcept {
    return Impl_->AllocatedBytes() + sizeof(*this);
}

template class TStringPoolBase<void>;
template class TStringPoolBase<TLightRWLock>;


TRCUStringPool::TRCUStringPool()
    : Impl_(new TImpl)
{
}

TRCUStringPool::~TRCUStringPool() {
}

TStringId TRCUStringPool::Intern(TStringBuf str) {
    return Impl_->Intern(str);
}

TStringBuf TRCUStringPool::Find(TStringId id) const noexcept {
    return Impl_->Find(id);
}

TStringId TRCUStringPool::Find(TStringBuf str) const noexcept {
    return Impl_->Find(str);
}

size_t TRCUStringPool::Size() const noexcept {
    return Impl_->Size();
}

void TRCUStringPool::Update() {
    Impl_->Update();
}

size_t TRCUStringPool::AllocatedBytes() const noexcept {
    return Impl_->AllocatedBytes();
}

} // namespace NSolomon::NIntern
