#pragma once

#include <util/generic/ptr.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <type_traits>

namespace NSolomon::NIntern {

using TStringId = ui32;
class TRCUStringPool;

static constexpr TStringId InvalidStringId = 0;

/**
 * Pool to replace string objects by 4-bytes unique identificators.
 * This pool uses block allocator to avoid memory fragmentation and RB-tree
 * for indexing to avoid blocks reallocation.
 *
 * All operations have O(log(n)) complexity, except Find(TStringId) which has O(1) complexity.
 * TStringPool is not thread-safe.
 * TConcurrentStringPool is thread-safe.
 */
template <typename TLock>
class TStringPoolBase {
public:
    TStringPoolBase();
    ~TStringPoolBase();

    friend class TRCUStringPool;

    /**
     * Find or add string to the pool.
     *
     * @return unique identificator of stored string.
     * @throw yexception if some error occured
     */
    TStringId Intern(TStringBuf str);

    /**
     * Find string by given identificator.
     *
     * @return valid string buffer if string was found by id,
     *         empty string buffer otherwise.
     */
    TStringBuf Find(TStringId id) const noexcept;

    /**
     * Find identificator by given string without pool modification.
     *
     * @return unique identificator if string was found in the pool
     *         {@code InvalidStringId} otherwise.
     */
    TStringId Find(TStringBuf str) const noexcept;

    /**
     * @return number of stored strings.
     */
    size_t Size() const noexcept;

    /**
     * @return how many bytes were allocated for this pool.
     */
    size_t AllocatedBytes() const noexcept;

    static_assert(std::is_same_v<void, TLock> || std::is_same_v<TLightRWLock, TLock>,
                  "only TStringPoolBase<void> and TStringPoolBase<TLightRwLock> instantiations are allowed");

private:
    struct TImpl;

    THolder<TImpl> Impl_;
};

class TRCUStringPool {
public:
    TRCUStringPool();
    ~TRCUStringPool();

    TStringId Intern(TStringBuf str);
    TStringBuf Find(TStringId id) const noexcept;
    TStringId Find(TStringBuf str) const noexcept;
    size_t Size() const noexcept;
    size_t AllocatedBytes() const noexcept;

    /**
     * update read only part of string pool
     */
     void Update();

private:
    struct TImpl;
    THolder<TImpl> Impl_;
};


using TStringPool = TStringPoolBase<void>;

using TConcurrentStringPool = TStringPoolBase<TLightRWLock>;

} // namespace NSolomon::NIntern
