#pragma once

#include "arena.h"

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>

#include <cstring>

namespace NSolomon::NIntern {

/**
 * Data layout:
 *
 * +-------------+
 * | const char* |---.
 * +-------------+   |
 *                   V
 *            +------+-------------------+----+
 *            | size |     string data   | \0 |
 *            +------+-------------------+----+
 */
class TStringRef {
    using TSize = ui16;

private:
    TStringRef(const char* data)
        : Data_(data)
    {
    }

public:
    /**
     * Copy string to arena and return a reference to it. Returns string ref and offset.
     */
    static std::pair<TStringRef, size_t> CreateFromString(TArena& arena, TStringBuf str) {
        Y_ENSURE(str.size() <= Max<TSize>(), "too long string, max size=" << Max<TSize>());

        size_t bytes = sizeof(TSize) + str.size() + 1;
        size_t alignment = alignof(TSize);
        char* strCopy = static_cast<char*>(arena.Alloc(bytes, alignment));
        Y_ENSURE(strCopy != nullptr, "cannot allocate string");
        size_t offset = arena.Offset() - bytes;

        TSize size = static_cast<TSize>(str.size());
        memcpy(strCopy, &size, sizeof(TSize));
        strCopy += sizeof(TSize);

        memcpy(strCopy, str.data(), size);
        *(strCopy + size) = '\0';

        return {TStringRef{strCopy}, offset};
    }

    /**
     * Return reference to a string using its offset as returned by `CreateFromString` method.
     */
    static TStringRef CreateFromOffset(const TArena& arena, size_t offset) {
        auto data = arena.Ptr<const char*>(offset) + sizeof(TSize);
        Y_ENSURE(data, "invalid string offset: " << offset);
        return TStringRef{data};
    }

public:
    constexpr operator TStringBuf() const noexcept {
        return {Data_, Size()};
    }

    constexpr const char* Data() const noexcept {
        return Data_;
    }

    constexpr size_t Size() const noexcept {
        return *((const TSize*) Data_ - 1); // propperly aligned
    }

private:
    const char* Data_;
};

} // namespace NSolomon::NIntern
