#pragma once

#include "arena.h"

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>

#include <cstring>

namespace NSolomon::NIntern {

/**
 * Data layout:
 *
 * +-------------+
 * | const char* |---.
 * +-------------+   |
 *                   V
 *   +--------+------+-------------------+----+
 *   |RefCount| size |     string data   | \0 |
 *   +--------+------+-------------------+----+
 */

class TStringRefCount {
    using TStringSize = ui16;
    using TRefCount = ui32;

private:
    explicit TStringRefCount(const char* data = nullptr)
        : Data_(data)
    {
    }
public:
    static constexpr size_t NonStringSize() {
        return sizeof(TStringSize) + sizeof(TRefCount) + 1;
    }

    /*
     * Write string, string length and ref counter to memory
     */
    static TStringRefCount InitOffsetFromString(char* strCopy, TStringBuf str) {
        TRefCount refCount(0);
        memcpy(strCopy, &refCount, sizeof(TRefCount));
        strCopy += sizeof(TRefCount);

        TStringSize strSize(str.size());
        memcpy(strCopy, &strSize, sizeof(TStringSize));
        strCopy += sizeof(TStringSize);

        memcpy(strCopy, str.data(), strSize);
        *(strCopy + strSize) = '\0';

        return TStringRefCount{strCopy};
    }

    /**
     * Copy string to arena and return a reference to it. Returns string ref and offset.
     */
    static TStringRefCount CreateFromString(TArena& arena, TStringBuf str, size_t blockSize) {
        Y_ENSURE(sizeof(TRefCount) + sizeof(TStringSize) + str.size() + 1 <= blockSize, "too long string, max size=" << blockSize);
        size_t align = blockSize % 8 == 0 ? 8 : 4;
        Y_VERIFY(blockSize % align == 0, "Cannot align block size: %lu.", blockSize);
        char* strCopy = static_cast<char*>(arena.Alloc(blockSize, align));
        Y_ENSURE(strCopy != nullptr, "cannot allocate string");
        return InitOffsetFromString(strCopy, str);
    }

    /**
     * Return reference to a string using its offset as returned by `CreateFromString` method.
     */
    static TStringRefCount CreateFromOffset(const TArena& arena, size_t offset) {
        auto data = arena.Ptr<const char*>(offset) + sizeof(TRefCount) + sizeof(TStringSize);
        Y_ENSURE(data, "invalid string offset: " << offset);
        return TStringRefCount{data};
    }

    /*
     * Copy string str to Allocated memory in arena (strCopy)
     */
    static TStringRefCount UpdateOffset(const TArena& arena, size_t offset, TStringBuf strBuf) {
        auto strCopy = arena.Ptr<char*>(offset);
        Y_ENSURE(strCopy, "invalid string offset: " << offset);
        return InitOffsetFromString(strCopy, strBuf);
    }

public:
    constexpr operator TStringBuf() const noexcept {
        return {Data_, Size()};
    }

    constexpr const char* Data() const noexcept {
        return Data_;
    }

    constexpr size_t Size() const noexcept {
        return *((const TStringSize*) Data_ - 1); // propperly aligned
    }

private:
    const char* Data_;
};

} // namespace NSolomon::NIntern
