#include <solomon/libs/cpp/intern/arena.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/vector.h>

using namespace NSolomon::NIntern;


TEST(TArenaTest, Alloc) {
    TArena arena(32);
    EXPECT_EQ(arena.AllocatedBytes(), (32u + 24u) + 40u);

    // [0, 9]
    void* p1 = arena.Alloc(10);
    ASSERT_TRUE(p1);
    EXPECT_EQ(reinterpret_cast<uintptr_t>(p1) % PLATFORM_DATA_ALIGN, 0u);
    EXPECT_EQ(arena.Pages(), 1u);

    // [16, 25]
    void* p2 = arena.Alloc(10);
    ASSERT_TRUE(p2);
    EXPECT_EQ(reinterpret_cast<uintptr_t>(p2) % PLATFORM_DATA_ALIGN, 0u);
    EXPECT_EQ(arena.Pages(), 1u);

    EXPECT_EQ(reinterpret_cast<uintptr_t>(p2) - reinterpret_cast<uintptr_t>(p1), 16u);

    // [0, 31]
    void* p3 = arena.Alloc(32);
    ASSERT_TRUE(p3);
    EXPECT_EQ(reinterpret_cast<uintptr_t>(p3) % PLATFORM_DATA_ALIGN, 0u);
    EXPECT_EQ(arena.Pages(), 2u);

    // not enaugh space in one page
    void* p4 = arena.Alloc(33);
    ASSERT_FALSE(p4);
    EXPECT_EQ(arena.Pages(), 2u);

    EXPECT_EQ(arena.AllocatedBytes(), 2u * (32u + 24u) + 40u);
}

TEST(TArenaTest, PutGet) {
    for (int pageSize: std::initializer_list<int>{16, 23, 32, 37}) {
        TArena arena(pageSize);
        TVector<int*> values;

        for (int i = 0; i < 1000; ++i) {
            int* ptr = arena.Put<int>(i + 1);
            ASSERT_TRUE(ptr);
            values.push_back(ptr);
            EXPECT_EQ(*ptr, i + 1);
        }

        for (int i = 0; i < 1000; ++i) {
            int* ptr = values[i];
            int value = arena.Get<int>(i);
            EXPECT_EQ(*ptr, value);
            EXPECT_EQ(value, i + 1);
        }
    }
}

TEST(TArenaTest, New) {
    struct TGeoPoint {
        ui32 Lat;
        ui32 Lon;

        TGeoPoint(ui32 lat, ui32 lon)
           : Lat(lat)
            , Lon(lon)
        {
        }
    };

    struct TTsPoint {
        ui64 Ts;
        ui64 Value;

        TTsPoint(ui64 ts, ui64 value)
            : Ts(ts)
            , Value(value)
        {
        }
    };

    TArena arena(32);

    auto* geoPoint = arena.New<TGeoPoint>(5545, 3737);
    ASSERT_TRUE(geoPoint);
    EXPECT_EQ(geoPoint->Lat, 5545u);
    EXPECT_EQ(geoPoint->Lon, 3737u);

    auto* tsPoint = arena.New<TTsPoint>(1569890000, 31415926535);
    ASSERT_TRUE(tsPoint);
    EXPECT_EQ(tsPoint->Ts, 1569890000u);
    EXPECT_EQ(tsPoint->Value, 31415926535u);
}
