#include <solomon/libs/cpp/intern/str_pool_hash_map.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <util/string/printf.h>

using namespace NSolomon::NIntern;

TEST(THashMapStringPoolTest, Intern) {
    THashMapStringPool pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    EXPECT_NE(one, InvalidStringId);
    EXPECT_NE(two, InvalidStringId);
    EXPECT_NE(one, two);

    auto uno = pool.Intern("one");
    auto dos = pool.Intern("two");

    EXPECT_EQ(one, uno);
    EXPECT_EQ(two, dos);
}

TEST(THashMapStringPoolTest, FindById) {
    THashMapStringPool pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    EXPECT_EQ("one", pool.Find(one));
    EXPECT_EQ("two", pool.Find(two));

    EXPECT_EQ(TStringBuf{}, pool.Find(two + 1));
}

TEST(THashMapStringPoolTest, FindByStr) {
    THashMapStringPool pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    EXPECT_EQ(one, pool.Find("one"));
    EXPECT_EQ(two, pool.Find("two"));
    EXPECT_EQ(InvalidStringId, pool.Find("three"));
}

TEST(THashMapStringPoolTest, AllocatedBytes) {
    THashMapStringPool pool;

    const ui32 count = 10000;
    for (ui32 i = 0; i < count; i++) {
        pool.Intern(Sprintf("0x%05d", i));
    }

    size_t totalSize = pool.AllocatedBytes();
    size_t strSize = Sprintf("0x%05d", 0).Size() + 1;

    Cerr << "size total: " << totalSize
         << ", string size: " << double(totalSize) / count
         << ", overhead total: " << (totalSize - strSize * count)
         << ", string overhead: " << double(totalSize - strSize * count) / count
         << Endl;

    //
    // estemated overhead = sizeof(TSize) + sizeof(TRbTreeNode) + sizeof(offset) + arenas overhead
    //                        2 bytes            48 bytes            8 bytes          ~X bytes
    //
    // estemated overhead ~ 58+X bytes
    //

    double perStringOverhead = double(totalSize - strSize * count) / count;
    EXPECT_LE(perStringOverhead, count * 10.0);
}

TEST(THashMapStringPoolTest, RemoveStr) {
    THashMapStringPool pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    pool.Remove(one);

    EXPECT_EQ(pool.Size(), 1u);

    EXPECT_EQ(pool.Find(two), "two");
    EXPECT_EQ(pool.Find(one), TStringBuf{});
    EXPECT_EQ(pool.Find("one"), InvalidStringId);
}

TEST(THashMapStringPoolTest, DecreaseCount) {
    THashMapStringPool pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");
    auto second_one = pool.Intern("one");

    EXPECT_EQ(one, second_one);

    pool.Remove(one);

    EXPECT_EQ(pool.Size(), 2u);

    EXPECT_EQ(pool.Find(two), "two");
    EXPECT_EQ(pool.Find(one), "one");
    EXPECT_EQ(pool.Find("one"), one);

    pool.Remove(one);
    EXPECT_EQ(pool.Size(), 1u);
}

TEST(THashMapStringPoolTest, RemoveStrings) {
    THashMapStringPool pool;
    auto secondString = "two hundred and forty-three thousand seven hundred and thirty-three";

    auto one = pool.Intern("one");
    auto two = pool.Intern(secondString);

    EXPECT_NE(one, InvalidStringId);
    EXPECT_NE(two, InvalidStringId);

    EXPECT_EQ(pool.Size(), 2u);

    EXPECT_EQ(pool.Intern(secondString), two);
    EXPECT_EQ(pool.Size(), 2u);

    pool.Remove(two);
    EXPECT_EQ(pool.Size(), 2u);

    pool.Remove(two);
    EXPECT_EQ(pool.Size(), 1u);
}
