#include <solomon/libs/cpp/intern/str_pool.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <util/string/printf.h>

using namespace NSolomon::NIntern;

template <typename T>
class TStringPoolTest: public ::testing::Test {
public:
    using TStringPoolImpl = T;
};

using TStringPoolTypes = ::testing::Types<TStringPool, TConcurrentStringPool, TRCUStringPool>;
TYPED_TEST_SUITE(TStringPoolTest, TStringPoolTypes);

TYPED_TEST(TStringPoolTest, Intern) {
    typename TestFixture::TStringPoolImpl pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    EXPECT_EQ(one, 1u);
    EXPECT_EQ(two, 2u);

    auto uno = pool.Intern("one");
    auto dos = pool.Intern("two");

    EXPECT_EQ(one, uno);
    EXPECT_EQ(two, dos);
}

TYPED_TEST(TStringPoolTest, FindById) {
    typename TestFixture::TStringPoolImpl pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    EXPECT_EQ(one, 1u);
    EXPECT_EQ(two, 2u);

    EXPECT_EQ("one", pool.Find(one));
    EXPECT_EQ("two", pool.Find(two));

    EXPECT_EQ(TStringBuf{}, pool.Find(two + 1));
}

TYPED_TEST(TStringPoolTest, FindByStr) {
    typename TestFixture::TStringPoolImpl pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    EXPECT_EQ(one, 1u);
    EXPECT_EQ(two, 2u);

    EXPECT_EQ(one, pool.Find("one"));
    EXPECT_EQ(two, pool.Find("two"));
    EXPECT_EQ(InvalidStringId, pool.Find("three"));
}

TEST(TRCUStringPool, Update) {
    TRCUStringPool pool;

    auto one = pool.Intern("one");
    auto two = pool.Intern("two");

    pool.Update();

    EXPECT_EQ(one, 1u);
    EXPECT_EQ(two, 2u);

    auto uno = pool.Intern("one");
    auto dos = pool.Intern("two");

    EXPECT_EQ(one, uno);
    EXPECT_EQ(two, dos);
}

TEST(TStringPoolTest, AllocatedBytes) {
    TStringPool pool;

    const ui32 count = 10000;
    for (ui32 i = 0; i < count; i++) {
        pool.Intern(Sprintf("0x%05d", i));
    }

    size_t totalSize = pool.AllocatedBytes();
    size_t strSize = Sprintf("0x%05d", 0).Size() + 1;

    Cerr << "size total: " << totalSize
         << ", string size: " << double(totalSize) / count
         << ", overhead total: " << (totalSize - strSize * count)
         << ", string overhead: " << double(totalSize - strSize * count) / count
         << Endl;

    //
    // estemated overhead = sizeof(TSize) + sizeof(TRbTreeNode) + sizeof(offset) + arenas overhead
    //                        2 bytes            48 bytes            8 bytes          ~X bytes
    //
    // estemated overhead ~ 58+X bytes
    //

    double perStringOverhead = double(totalSize - strSize * count) / count;
    EXPECT_LE(perStringOverhead, count * 10.0);
}
