#include "actor_bridge.h"

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/fastqueue.h>
#include <util/stream/output.h>

#include <utility>

namespace NSolomon::NKv {

namespace {

/**
 * This class contains data about a single request.
 */
class TRequestCtx: public TThrRefBase {
public:
    explicit TRequestCtx(bool allowRetry = true)
        : NumRetries(allowRetry ? 0 : -1)
    {
    }

    /**
     * Send a new request and return a future that resolves whenever a response is received.
     * Should not increment `NumRetries`.
     */
    virtual NThreading::TFuture<void> Request(TKikimrKvClient&) = 0;

    /**
     * If the last request finished with an error, should return pointer to said error.
     * If there were no requests or if they finished successfully, should return `nullptr`.
     */
    virtual const TKvClientError* MaybeGetError() const = 0;

    /**
     * Create an event that contains result of the latest request sent from this context.
     * This function should panic if `Request` has never been called.
     */
    virtual THolder<NActors::IEventBase> MakeReply() = 0;

    /**
     * Create an event that contains the given error.
     * This function should work if `Request` has never been called.
     */
    virtual THolder<NActors::IEventBase> MakeError(TKvClientError error) = 0;

    /**
     * Get human-readable name of this request.
     */
    virtual TString What() const = 0;

    /**
     * How many requests have been sent from this context so far.
     */
    int NumRetries = 0;

    /**
     * Who should get a response generated by `MakeReply` or `MakeError`.
     */
    NActors::TActorId ReplyTo;

    /**
     * Cookie to send with reply.
     */
    ui64 Cookie = 0;
};

using TRequestCtxPtr = TIntrusivePtr<TRequestCtx>;

template <typename TEvent_>
class TRequestCtxImpl: public TRequestCtx {
public:
    using TRequestCtx::TRequestCtx;

    using TEvent = TEvent_;
    using TResult = typename TEvent::TResult;

    NThreading::TFuture<void> HandleResult(TAsyncKvResult<TResult> future) {
        auto ptr = TIntrusivePtr<TRequestCtx>(this);
        return future
            .Subscribe([this, ptr](TAsyncKvResult<TResult> future) {
                Y_UNUSED(ptr);
                Result_ = future.ExtractValueSync();
            })
            .IgnoreResult();
    }

    const TKvClientError* MaybeGetError() const override {
        if (Result_.Fail()) {
            return &Result_.Error();
        } else {
            return nullptr;
        }
    }

    THolder<NActors::IEventBase> MakeReply() override {
        return MakeHolder<TEvent>(std::move(Result_));
    }

    THolder<NActors::IEventBase> MakeError(TKvClientError error) override {
        return MakeHolder<TEvent>(std::move(error));
    }

private:
    TKvResult<TResult> Result_ = TKvClientError(grpc::UNKNOWN);
};

} // namespace

struct TEvents::TRequest: public NActors::TEventLocal<TRequest, Request> {
    TRequestCtxPtr Request;

    explicit TRequest(TRequestCtxPtr request)
        : Request(std::move(request))
    {
    }
};

namespace {

class TCreateSolomonVolumeRequestCtx: public TRequestCtxImpl<TEvents::TCreateSolomonVolumeResponse> {
public:
    explicit TCreateSolomonVolumeRequestCtx(TString path, ui64 partitionCount, ui32 channelProfileId, bool retry)
        : TRequestCtxImpl(retry)
        , Path_(std::move(path))
        , PartitionCount_(partitionCount)
        , ChannelProfileId_(channelProfileId)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.CreateSolomonVolume(Path_, PartitionCount_, ChannelProfileId_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "CreateSolomonVolume{"
            << "path=" << Path_ << ", "
            << "partitionCount=" << PartitionCount_ << ", "
            << "channelProfileId=" << ChannelProfileId_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    TString Path_;
    ui64 PartitionCount_;
    ui32 ChannelProfileId_;
};

class TDropSolomonVolumeRequestCtx: public TRequestCtxImpl<TEvents::TDropSolomonVolumeResponse> {
public:
    explicit TDropSolomonVolumeRequestCtx(TString path, bool retry)
        : TRequestCtxImpl(retry)
        , Path_(std::move(path))
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.DropSolomonVolume(Path_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "DropSolomonVolume{"
            << "path=" << Path_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    TString Path_;
};

class TResolveTabletsRequestCtx: public TRequestCtxImpl<TEvents::TResolveTabletsResponse> {
public:
    explicit TResolveTabletsRequestCtx(TString solomonVolumePath, bool retry)
        : TRequestCtxImpl(retry)
        , SolomonVolumePath_(std::move(solomonVolumePath))
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.ResolveTablets(SolomonVolumePath_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "ResolveTablets{"
            << "path=" << SolomonVolumePath_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    TString SolomonVolumePath_;
};

class TResolveTabletsRequestCtxO2: public TRequestCtxImpl<TEvents::TResolveTabletsResponse> {
public:
    explicit TResolveTabletsRequestCtxO2(ui64 ownerId, TVector<ui64> ownerIdxs, bool retry)
        : TRequestCtxImpl(retry)
        , OwnerId_(ownerId)
        , OwnerIdxs_(std::move(ownerIdxs))
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.ResolveTablets(OwnerId_, OwnerIdxs_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "ResolveTablets{"
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 OwnerId_;
    TVector<ui64> OwnerIdxs_;
};

class TLocalTabletsRequestCtx: public TRequestCtxImpl<TEvents::TLocalTabletsResponse> {
public:
    explicit TLocalTabletsRequestCtx(bool retry)
        : TRequestCtxImpl(retry)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.LocalTablets());
    }

    TString What() const override {
        return TStringBuilder{}
            << "LocalTablets{"
            << "cookie=" << Cookie << "}";
    }
};

class TTabletsInfoRequestCtx: public TRequestCtxImpl<TEvents::TTabletsInfoResponse> {
public:
    explicit TTabletsInfoRequestCtx(TVector<ui64> tabletIds, bool retry)
        : TRequestCtxImpl(retry)
        , TabletIds_(std::move(tabletIds))
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.TabletsInfo(TabletIds_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "TabletsInfo{"
            << "cookie=" << Cookie << "}";
    }

private:
    TVector<ui64> TabletIds_;
};

class TIncrementGenerationRequestCtx: public TRequestCtxImpl<TEvents::TIncrementGenerationResponse> {
public:
    explicit TIncrementGenerationRequestCtx(ui64 tabletId, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.IncrementGeneration(TabletId_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "IncrementGeneration{"
            << "tabletId=" << TabletId_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TInstant Deadline_;
};

class TListFilesRequestCtx: public TRequestCtxImpl<TEvents::TListFilesResponse> {
public:
    explicit TListFilesRequestCtx(ui64 tabletId, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.ListFiles(TabletId_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "ListFiles{"
            << "tabletId=" << TabletId_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TInstant Deadline_;
};

class TReadFileRequestCtx: public TRequestCtxImpl<TEvents::TReadFileResponse> {
public:
    explicit TReadFileRequestCtx(ui64 tabletId, TString name, ui64 offset, ui64 size, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , Name_(std::move(name))
        , Offset_(offset)
        , Size_(size)
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.ReadFile(TabletId_, Name_, Offset_, Size_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "ReadFile{"
            << "tabletId=" << TabletId_ << ", "
            << "name=" << Name_ << ", "
            << "offset=" << Offset_ << ", "
            << "size=" << Size_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TString Name_;
    ui64 Offset_;
    ui64 Size_;
    TInstant Deadline_;
};

class TWriteFileRequestCtx: public TRequestCtxImpl<TEvents::TWriteFileResponse> {
public:
    explicit TWriteFileRequestCtx(ui64 tabletId, TString name, TString data, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , Name_(std::move(name))
        , Data_(std::move(data))
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.WriteFile(TabletId_, Name_, Data_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "WriteFile{"
            << "tabletId=" << TabletId_ << ", "
            << "name=" << Name_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TString Name_;
    TString Data_;
    TInstant Deadline_;
};

class TRenameFileRequestCtx: public TRequestCtxImpl<TEvents::TRenameFileResponse> {
public:
    explicit TRenameFileRequestCtx(ui64 tabletId, TString source, TString target, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , Source_(std::move(source))
        , Target_(std::move(target))
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.RenameFile(TabletId_, Source_, Target_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "RenameFile{"
            << "tabletId=" << TabletId_ << ", "
            << "source=" << Source_ << ", "
            << "target=" << Target_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TString Source_;
    TString Target_;
    TInstant Deadline_;
};

class TCopyFilesRequestCtx: public TRequestCtxImpl<TEvents::TCopyFilesResponse> {
public:
    explicit TCopyFilesRequestCtx(ui64 tabletId, TKikimrKvRange from, TString prefixToRemove, TString prefixToAdd, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , From_(std::move(from))
        , PrefixToRemove_(std::move(prefixToRemove))
        , PrefixToAdd_(std::move(prefixToAdd))
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.CopyFiles(TabletId_, From_, PrefixToRemove_, PrefixToAdd_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "CopyFiles{"
            << "tabletId=" << TabletId_ << ", "
            << "from=" << From_ << ", "
            << "prefixToRemove=" << PrefixToRemove_ << ", "
            << "prefixToAdd=" << PrefixToAdd_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TKikimrKvRange From_;
    TString PrefixToRemove_;
    TString PrefixToAdd_;
    TInstant Deadline_;
};

class TRemoveFilesRequestCtx: public TRequestCtxImpl<TEvents::TRemoveFilesResponse> {
public:
    explicit TRemoveFilesRequestCtx(ui64 tabletId, TKikimrKvRange file, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , File_(std::move(file))
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.RemoveFiles(TabletId_, File_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "RemoveFiles{"
            << "tabletId=" << TabletId_ << ", "
            << "file=" << File_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TKikimrKvRange File_;
    TInstant Deadline_;
};

class TRemovePrefixRequestCtx: public TRequestCtxImpl<TEvents::TRemovePrefixResponse> {
public:
    explicit TRemovePrefixRequestCtx(ui64 tabletId, TString prefix, TInstant deadline, bool retry)
        : TRequestCtxImpl(retry)
        , TabletId_(tabletId)
        , Prefix_(std::move(prefix))
        , Deadline_(deadline)
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.RemovePrefix(TabletId_, Prefix_, Deadline_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "RemovePrefix{"
            << "tabletId=" << TabletId_ << ", "
            << "prefix=" << Prefix_ << ", "
            << "cookie=" << Cookie << "}";
    }

private:
    ui64 TabletId_;
    TString Prefix_;
    TInstant Deadline_;
};

class TBatchRequestCtx: public TRequestCtxImpl<TEvents::TBatchResponse> {
public:
    TBatchRequestCtx(TKikimrKvBatchRequest batchRequest, bool retry)
        : TRequestCtxImpl(retry)
        , BatchRequest_(std::move(batchRequest))
    {
    }

    NThreading::TFuture<void> Request(TKikimrKvClient& client) override {
        return HandleResult(client.BatchRequest(BatchRequest_));
    }

    TString What() const override {
        return TStringBuilder{}
            << "BatchRequest{"
            << "cookie=" << Cookie << "}";
    }

private:
    TKikimrKvBatchRequest BatchRequest_;
};

} // namespace

THolder<NActors::IEventBase> TEvents::CreateSolomonVolume(TString path, ui64 partitionCount, ui32 channelProfileId, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TCreateSolomonVolumeRequestCtx(std::move(path), partitionCount, channelProfileId, retry));
}

THolder<NActors::IEventBase> TEvents::DropSolomonVolume(TString path, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TDropSolomonVolumeRequestCtx(std::move(path), retry));
}

THolder<NActors::IEventBase> TEvents::ResolveTablets(TString solomonVolumePath, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TResolveTabletsRequestCtx(std::move(solomonVolumePath), retry));
}

THolder<NActors::IEventBase> TEvents::ResolveTablets(ui64 ownerId, TVector<ui64> ownerIdxs, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TResolveTabletsRequestCtxO2(ownerId, std::move(ownerIdxs), retry));
}

THolder<NActors::IEventBase> TEvents::LocalTablets(bool retry) {
    return MakeHolder<TEvents::TRequest>(new TLocalTabletsRequestCtx(retry));
}

THolder<NActors::IEventBase> TEvents::TabletsInfo(TVector<ui64> tabletIds, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TTabletsInfoRequestCtx(std::move(tabletIds), retry));
}

THolder<NActors::IEventBase> TEvents::IncrementGeneration(ui64 tabletId, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TIncrementGenerationRequestCtx(tabletId, deadline, retry));
}

THolder<NActors::IEventBase> TEvents::ListFiles(ui64 tabletId, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TListFilesRequestCtx(tabletId, deadline, retry));
}

THolder<NActors::IEventBase> TEvents::ReadFile(ui64 tabletId, TString name, ui64 offset, ui64 size, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TReadFileRequestCtx(tabletId, std::move(name), offset, size, deadline, retry));
}

THolder<NActors::IEventBase> TEvents::WriteFile(ui64 tabletId, TString name, TString data, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TWriteFileRequestCtx(tabletId, std::move(name), std::move(data), deadline, retry));
}

THolder<NActors::IEventBase> TEvents::RenameFile(ui64 tabletId, TString source, TString target, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TRenameFileRequestCtx(tabletId, std::move(source), std::move(target), deadline, retry));
}

THolder<NActors::IEventBase> TEvents::CopyFile(ui64 tabletId, const TString& from, TString to, TInstant deadline, bool retry) {
    return CopyFiles(tabletId, TKikimrKvRange{from, true, from, true}, from, std::move(to), deadline, retry);
}

THolder<NActors::IEventBase> TEvents::CopyFiles(ui64 tabletId, TKikimrKvRange from, TString prefixToRemove, TString prefixToAdd, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TCopyFilesRequestCtx(tabletId, std::move(from), std::move(prefixToRemove), std::move(prefixToAdd), deadline, retry));
}

THolder<NActors::IEventBase> TEvents::RemoveFile(ui64 tabletId, const TString& file, TInstant deadline, bool retry) {
    return RemoveFiles(tabletId, TKikimrKvRange{file, true, file, true}, deadline, retry);
}

THolder<NActors::IEventBase> TEvents::RemoveFiles(ui64 tabletId, TKikimrKvRange file, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TRemoveFilesRequestCtx(tabletId, std::move(file), deadline, retry));
}

THolder<NActors::IEventBase> TEvents::RemovePrefix(ui64 tabletId, TString prefix, TInstant deadline, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TRemovePrefixRequestCtx(tabletId, std::move(prefix), deadline, retry));
}

THolder<NActors::IEventBase> TEvents::Batch(TKikimrKvBatchRequest batchRequest, bool retry) {
    return MakeHolder<TEvents::TRequest>(new TBatchRequestCtx(std::move(batchRequest), retry));
}

namespace {

class TKvClientActor: public NActors::TActor<TKvClientActor>, private TPrivateEvents {
#define LOG_P "{kv client " << SelfId() << " (" << GetAddress() << ")} "

    enum {
        Response = SpaceBegin,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

    struct TResponse: public NActors::TEventLocal<TResponse, Response> {
        TRequestCtxPtr Request;

        explicit TResponse(TRequestCtxPtr request)
            : Request(std::move(request)) {
        }
    };

public:
    TKvClientActor(TString address, NKikimr::IKikimrRpc* nodeRpc, std::shared_ptr<NKikimr::IKikimrClusterRpc> rpc, TRetryOpts retryOpts)
        : TActor(&TKvClientActor::Main)
        , Address_{std::move(address)}
        , Client_{nodeRpc}
        , Rpc_{std::move(rpc)}
        , RetryOpts_{std::move(retryOpts)}
    {
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::TCommonEvents::TAsyncPoison, OnAsyncPoison);
            hFunc(TEvents::TRequest, OnRequest);
            hFunc(TResponse, OnResponse);
        }
    }

    void OnAsyncPoison(typename NSolomon::TCommonEvents::TAsyncPoison::TPtr& ev) {
        if (InFlight_ == 0) {
            PassAway();
            ev->Get()->Done();
        } else {
            PoisonEvent_ = ev->Release();
            Become(&TKvClientActor::Closed);
        }
    }

    void OnRequest(typename TEvents::TRequest::TPtr& ev) {
        auto request = std::move(ev->Get()->Request);
        SetupRequest(ev, request);

        MON_DEBUG(KvClient, LOG_P << "sending request for " << *request);
        auto system = NActors::TActorContext::ActorSystem();
        auto id = SelfId();
        request->Request(Client_)
            .Subscribe([request = std::move(request), system, id](const NThreading::TFuture<void>&) {
                system->Send(id, new TResponse(request));
            });
    }

    void OnResponse(typename TResponse::TPtr& ev) {
        auto request = std::move(ev->Get()->Request);

        auto* error = request->MaybeGetError();
        if (IsTransient(error)) {
            if (CanRetry(request.Get())) {
                MON_ERROR(KvClient, LOG_P << *request << " finished with transient error " << error->Message() << ", "
                                          << "will be retried");
                Retry(std::move(request));
                return;
            } else {
                MON_ERROR(KvClient, LOG_P << *request << " finished with transient error " << error->Message());
            }
        } else if (error) {
            MON_ERROR(KvClient, LOG_P << *request << " finished with non-transient error " << error->Message());
        } else {
            MON_DEBUG(KvClient, LOG_P << *request << " finished with success");
        }

        MON_DEBUG(KvClient, LOG_P << "sending response for " << *request);
        Send(
            request->ReplyTo,
            request->MakeReply(),
            /* flags = */ 0,
            /* cookie = */ request->Cookie);
        InFlight_ -= 1;
    }

    bool IsTransient(const TKvClientError* error) {
        return error && (
            error->IsTransportError() && RetryOpts_.RetryCodes.contains(error->AsTransportError()) ||
            error->IsKvError() && RetryOpts_.KvRetryCodes.contains(error->AsKvError()));
    }

    bool CanRetry(TRequestCtx* request) {
        return request->NumRetries != -1 && (!RetryOpts_.MaxRetries || request->NumRetries < RetryOpts_.MaxRetries);
    }

    void Retry(TRequestCtxPtr request) {
        auto backoffTime = RetryOpts_.BackoffTime;
        for (int i = 0; i < request->NumRetries; ++i) {
            backoffTime *= RetryOpts_.BackoffFactor;
            if (backoffTime > RetryOpts_.BackoffTimeMax) {
                backoffTime = RetryOpts_.BackoffTimeMax;
                break;
            }
        }
        request->NumRetries += 1;

        MON_DEBUG(KvClient,
            LOG_P << "scheduling retry for " << *request
                  << ": num_retries=" << request->NumRetries << ", backoff_time=" << backoffTime);
        Schedule(backoffTime, new TEvents::TRequest(std::move(request)));
    }

    TStringBuf GetAddress() const {
        return Address_;
    }

    STATEFN(Closed) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TRequest, OnRequestClosed);
            hFunc(TResponse, OnResponseClosed);
        }
    }

    void OnRequestClosed(typename TEvents::TRequest::TPtr& ev) {
        auto request = std::move(ev->Get()->Request);
        SetupRequest(ev, request);

        MON_INFO(KvClient, LOG_P << "client is closed, dropping request " << *request);
        Send(
            request->ReplyTo,
            request->MakeError(TKvClientError{grpc::UNAVAILABLE, "client is closed"}),
            /* flags = */ 0,
            /* cookie = */ request->Cookie);
        InFlight_ -= 1;
        MaybePassAway();
    }

    void OnResponseClosed(typename TResponse::TPtr& ev) {
        auto request = std::move(ev->Get()->Request);

        MON_INFO(KvClient, LOG_P << "client is closed, sending response for " << *request << " as is");
        Send(
            request->ReplyTo,
            request->MakeReply(),
            /* flags = */ 0,
            /* cookie = */ request->Cookie);
        InFlight_ -= 1;
        MaybePassAway();
    }

    void SetupRequest(const NActors::TEventBase<TEvents::TRequest, 0>::TPtr& ev, TIntrusivePtr<TRequestCtx>& request) {
        if (!request->ReplyTo) {
            MON_DEBUG(KvClient, LOG_P << "new request " << *request);
            request->ReplyTo = ev->Sender;
            request->Cookie = ev->Cookie;
            InFlight_ += 1;
        } else {
            MON_DEBUG(KvClient, LOG_P << "retrying request " << *request);
        }
    }

    void MaybePassAway() {
        if (InFlight_ == 0) {
            PassAway();
            if (PoisonEvent_) {
                PoisonEvent_->Done();
            }
        }
    }

private:
    const TString Address_;
    TKikimrKvClient Client_;
    std::shared_ptr<NKikimr::IKikimrClusterRpc> Rpc_;
    TRetryOpts RetryOpts_;
    size_t InFlight_ = 0;
    THolder<TCommonEvents::TAsyncPoison> PoisonEvent_;
};

} // namespace

THolder<NActors::IActor> CreateKvClientActor(TString address, NKikimr::IKikimrRpc* nodeRpc, TRetryOpts retryOpts) {
    return MakeHolder<TKvClientActor>(std::move(address), nodeRpc, nullptr, std::move(retryOpts));
}

THolder<NActors::IActor> CreateKvClientActor(TString address, std::shared_ptr<NKikimr::IKikimrClusterRpc> rpc, TRetryOpts retryOpts) {
    auto* nodeRpc = rpc->Get(address);
    return MakeHolder<TKvClientActor>(std::move(address), nodeRpc, std::move(rpc), std::move(retryOpts));
}

} // namespace NSolomon::NKv

Y_DECLARE_OUT_SPEC(, NSolomon::NKv::TRequestCtx, out, t) {
    out << t.What() << "#" << static_cast<const void*>(&t);
}
