#pragma once

/**
 * An actor that provides an interface for YDB KV storage.
 */

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/events/slots.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/kv/kv_client.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NKv {

class TEvents: private TEventSlot<EEventSpace::Libs, ELibSlot::KvClient> {
    enum {
        Request = SpaceBegin,
        CreateSolomonVolumeResponse,
        DropSolomonVolumeResponse,
        ResolveTabletsResponse,
        LocalTabletsResponse,
        TabletsInfoResponse,
        IncrementGenerationResponse,
        ListFilesResponse,
        ReadFileResponse,
        WriteFileResponse,
        RenameFileResponse,
        CopyFilesResponse,
        RemoveFilesResponse,
        RemovePrefixResponse,
        BatchResponse,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    static
    bool IsKvEvent(ui32 eventType) {
        return Request <= eventType && eventType < SpaceEnd;
    }

    struct TRequest;

    struct TCreateSolomonVolumeResponse
        : public NActors::TEventLocal<TCreateSolomonVolumeResponse, CreateSolomonVolumeResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TCreateSolomonVolumeResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TDropSolomonVolumeResponse
        : public NActors::TEventLocal<TDropSolomonVolumeResponse, DropSolomonVolumeResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TDropSolomonVolumeResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TResolveTabletsResponse
        : public NActors::TEventLocal<TResolveTabletsResponse, ResolveTabletsResponse>
        , public TErrorOr<TVector<ui64>, TKvClientError>
    {
        using TResult = TVector<ui64>;
        explicit TResolveTabletsResponse(TErrorOr<TVector<ui64>, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TLocalTabletsResponse
        : public NActors::TEventLocal<TLocalTabletsResponse, LocalTabletsResponse>
        , public TErrorOr<TVector<ui64>, TKvClientError>
    {
        using TResult = TVector<ui64>;
        explicit TLocalTabletsResponse(TErrorOr<TVector<ui64>, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TTabletsInfoResponse
        : public NActors::TEventLocal<TTabletsInfoResponse, TabletsInfoResponse>
        , public TErrorOr<TVector<TTabletInfo>, TKvClientError>
    {
        using TResult = TVector<TTabletInfo>;
        explicit TTabletsInfoResponse(TErrorOr<TVector<TTabletInfo>, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TIncrementGenerationResponse
        : public NActors::TEventLocal<TIncrementGenerationResponse, IncrementGenerationResponse>
        , public TErrorOr<ui64, TKvClientError>
    {
        using TResult = ui64;
        explicit TIncrementGenerationResponse(TErrorOr<ui64, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TListFilesResponse
        : public NActors::TEventLocal<TListFilesResponse, ListFilesResponse>
        , public TErrorOr<TVector<TFileInfo>, TKvClientError>
    {
        using TResult = TVector<TFileInfo>;
        explicit TListFilesResponse(TErrorOr<TVector<TFileInfo>, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TReadFileResponse
        : public NActors::TEventLocal<TReadFileResponse, ReadFileResponse>
        , public TErrorOr<TString, TKvClientError>
    {
        using TResult = TString;
        explicit TReadFileResponse(TErrorOr<TString, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TWriteFileResponse
        : public NActors::TEventLocal<TWriteFileResponse, WriteFileResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TWriteFileResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TRenameFileResponse
        : public NActors::TEventLocal<TRenameFileResponse, RenameFileResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TRenameFileResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TCopyFilesResponse
        : public NActors::TEventLocal<TCopyFilesResponse, CopyFilesResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TCopyFilesResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TRemoveFilesResponse
        : public NActors::TEventLocal<TRemoveFilesResponse, RemoveFilesResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TRemoveFilesResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TRemovePrefixResponse
        : public NActors::TEventLocal<TRemovePrefixResponse, RemovePrefixResponse>
        , public TErrorOr<void, TKvClientError>
    {
        using TResult = void;
        explicit TRemovePrefixResponse(TErrorOr<void, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

    struct TBatchResponse
        : public NActors::TEventLocal<TBatchResponse, BatchResponse>
        , public TErrorOr<TKikimrKvBatchResult, TKvClientError>
    {
        using TResult = TKikimrKvBatchResult;
        explicit TBatchResponse(TErrorOr<TKikimrKvBatchResult, TKvClientError> result) : TErrorOr(std::move(result)) {}
    };

public:
    static THolder<NActors::IEventBase> CreateSolomonVolume(
        TString path,
        ui64 partitionCount,
        ui32 channelProfileId = 0,
        bool retry = false);

    static THolder<NActors::IEventBase> DropSolomonVolume(
        TString path,
        bool retry = false);

    static THolder<NActors::IEventBase> ResolveTablets(
        TString solomonVolumePath,
        bool retry = true);

    static THolder<NActors::IEventBase> ResolveTablets(
        ui64 ownerId,
        TVector<ui64> ownerIdxs,
        bool retry = true);

    static THolder<NActors::IEventBase> LocalTablets(
        bool retry = true);

    static THolder<NActors::IEventBase> TabletsInfo(
        TVector<ui64> tabletIds,
        bool retry = true);

    static THolder<NActors::IEventBase> IncrementGeneration(
        ui64 tabletId,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> ListFiles(
        ui64 tabletId,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = true);

    static THolder<NActors::IEventBase> ReadFile(
        ui64 tabletId,
        TString name,
        ui64 offset = 0,
        ui64 size = 0,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = true);

    static THolder<NActors::IEventBase> WriteFile(
        ui64 tabletId,
        TString name,
        TString data,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> RenameFile(
        ui64 tabletId,
        TString source,
        TString target,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> CopyFile(
        ui64 tabletId,
        const TString& from,
        TString to,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> CopyFiles(
        ui64 tabletId,
        TKikimrKvRange from,
        TString prefixToRemove,
        TString prefixToAdd,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> RemoveFile(
        ui64 tabletId,
        const TString& file,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> RemoveFiles(
        ui64 tabletId,
        TKikimrKvRange file,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> RemovePrefix(
        ui64 tabletId,
        TString prefix,
        TInstant delay = TDuration::Seconds(60).ToDeadLine(),
        bool retry = false);

    static THolder<NActors::IEventBase> Batch(
        TKikimrKvBatchRequest request,
        bool retry = false);
};

struct TRetryOpts {
    /**
     * Maximum number of retries for each request. `0` means unlimited.
     */
    int MaxRetries = 4;

    /**
     * If some request failed, send next request no sooner than this time.
     */
    TDuration BackoffTime = TDuration::MilliSeconds(100);

    /**
     * If some request failed, send next request no later than this time.
     */
    TDuration BackoffTimeMax = TDuration::MilliSeconds(800);

    /**
     * With each retry, multiply delay before next retry by this number until it reaches `BackoffTimeMax`.
     */
    float BackoffFactor = 2;

    /**
     * Which transport errors should be retried.
     */
    THashSet<grpc::StatusCode> RetryCodes = {
        grpc::DEADLINE_EXCEEDED,
        grpc::RESOURCE_EXHAUSTED,
        grpc::INTERNAL,
        grpc::UNAVAILABLE,
    };

    /**
     * Which KV errors should be retried.
     */
    THashSet<::NKikimr::NMsgBusProxy::EResponseStatus> KvRetryCodes = {
        ::NKikimr::NMsgBusProxy::MSTATUS_TIMEOUT,
        ::NKikimr::NMsgBusProxy::MSTATUS_NOTREADY,
        ::NKikimr::NMsgBusProxy::MSTATUS_ABORTED,
        ::NKikimr::NMsgBusProxy::MSTATUS_INTERNALERROR,
        ::NKikimr::NMsgBusProxy::MSTATUS_REJECTED,
    };
};

/**
 * An actor that provides an interface for YDB KV storage. Implements retry strategy.
 *
 * # Supported events
 *
 * - **NSolomon::TCommonEvents::TAsyncPoison** -- wait for all requests to finish, then kill this client.
 *   After receiving this event, the client closes, meaning that all new requests will be rejected,
 *   and status updates will not be sent.
 * - **TEvents::TRequest** -- send new requests to YDB. Use static functions in `NSolomon::NKv::TEvents`
 *   to create instances of this event.
 */
THolder<NActors::IActor> CreateKvClientActor(
        TString address,
        NKikimr::IKikimrRpc* rpc,
        TRetryOpts retryOpts = {});

THolder<NActors::IActor> CreateKvClientActor(
        TString address,
        std::shared_ptr<NKikimr::IKikimrClusterRpc> rpc,
        TRetryOpts retryOpts = {});

} // namespace NSolomon::NKv
