#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/kv/actor_bridge/actor_bridge.h>
#include <solomon/libs/cpp/kv/testlib/testlib.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <util/system/env.h>

class KvClientActorTest: public NSolomon::NKv::TTestFixtureTablet<1> {
};

class KvClientActorFileTest: public KvClientActorTest {
public:
    void SetUp() override {
        TTestFixtureTablet::SetUp();

        Runtime = NSolomon::TTestActorRuntime::CreateInited();
        ClientActor = Runtime->Register(NSolomon::NKv::CreateKvClientActor(Endpoint, Rpc.get()).Release());
        ListenerActor = Runtime->AllocateEdgeActor();
    }

    void TearDown() override {
        Client.reset();
        TTestFixtureTablet::TearDown();
    }

    THolder<NSolomon::TTestActorRuntime> Runtime;
    NActors::TActorId ClientActor;
    NActors::TActorId ListenerActor;
};

TEST_F(KvClientActorFileTest, CreateAndDropSolomonVolume) {
    auto solomonVolumePath = SolomonVolumePath + "_2";

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::CreateSolomonVolume(solomonVolumePath, 2));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TCreateSolomonVolumeResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ResolveTablets(solomonVolumePath));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TResolveTabletsResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        auto tabletIds = response->Get()->Value();
        ASSERT_EQ(tabletIds.size(), 2u);

        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::LocalTablets());
        auto localTabletsResponse = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TLocalTabletsResponse>(ListenerActor);
        ASSERT_TRUE(localTabletsResponse->Get()->Success());
        auto localTabletIds = localTabletsResponse->Get()->Value();
        for (auto tabletId: tabletIds) {
            EXPECT_TRUE(Find(localTabletIds, tabletId) != localTabletIds.end());
        }
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::DropSolomonVolume(solomonVolumePath));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TDropSolomonVolumeResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ResolveTablets(solomonVolumePath));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TResolveTabletsResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Fail());
        EXPECT_EQ(response->Get()->Error().AsKvError(), NKikimr::NMsgBusProxy::MSTATUS_ERROR);
    }
}

TEST_F(KvClientActorFileTest, CreateSolomonVolumeError) {
    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::CreateSolomonVolume("_invalid_db_", 1));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TCreateSolomonVolumeResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Fail());
        EXPECT_EQ(response->Get()->Error().AsKvError(), NKikimr::NMsgBusProxy::MSTATUS_ERROR);
    }
}

TEST_F(KvClientActorFileTest, ListFiles) {
    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ListFiles(TabletId));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        auto files = response->Get()->Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "").GetValueSync();
    StableClient->WriteFile(TabletId, "b", "").GetValueSync();

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ListFiles(TabletId));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        auto files = response->Get()->Value();
        ASSERT_EQ(files.size(), 2u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(files[1].Name, "b");
    }
}

TEST_F(KvClientActorFileTest, ReadWriteFile) {
    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::WriteFile(TabletId, "a", "content"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TWriteFileResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ReadFile(TabletId, "a"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TReadFileResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        EXPECT_EQ(response->Get()->Value(), "content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ReadFile(TabletId, "b"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TReadFileResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        EXPECT_EQ(response->Get()->Value(), "");
    }
}

TEST_F(KvClientActorFileTest, RenameFile) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::RenameFile(TabletId, "a", "x"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TRenameFileResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[1].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[2].Name, "x");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x").GetValueSync().Value(), "a content");
    }
}

TEST_F(KvClientActorFileTest, CopyFiles) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::CopyFiles(TabletId, {"b", "b~"}, "b", "c"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TCopyFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 5u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[3].Name, "c1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[4].Name, "c2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c2").GetValueSync().Value(), "b2 content");
    }
}

TEST_F(KvClientActorFileTest, RemoveFiles) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();
    StableClient->WriteFile(TabletId, "c", "c content").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 4u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[3].Name, "c");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c").GetValueSync().Value(), "c content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::RemoveFile(TabletId, "c"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TRemoveFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::RemoveFile(TabletId, "b"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TRemoveFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::RemoveFiles(TabletId, {"b", "b~"}));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TRemoveFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 1u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
    }
}

TEST_F(KvClientActorFileTest, RemovePrefix) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::RemovePrefix(TabletId, "b"));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TRemovePrefixResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[1].Name, "2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[2].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
    }
}

TEST_F(KvClientActorFileTest, TabletOperationsBatch) {
    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::Batch(
                NSolomon::TKikimrKvBatchRequest(TabletId)
                    .WriteFile("l1", "log 1")
                    .WriteFile("l2", "log 2")
                    .WriteFile("l3", "log 3")));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TBatchResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    TVector<NSolomon::TFileInfo> files;
    files = StableClient->ListFiles(TabletId).GetValueSync().Value();
    ASSERT_EQ(files.size(), 3u);
    EXPECT_EQ(files[0].Name, "l1");
    EXPECT_EQ(files[1].Name, "l2");
    EXPECT_EQ(files[2].Name, "l3");

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::Batch(
                NSolomon::TKikimrKvBatchRequest(TabletId)
                    .WriteFile("s1", "snapshot 1")
                    .WriteFile("s2", "snapshot 2")
                    .RemoveFiles({"l", "l~"})));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TBatchResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
    }

    files = StableClient->ListFiles(TabletId).GetValueSync().Value();
    ASSERT_EQ(files.size(), 2u);
    EXPECT_EQ(files[0].Name, "s1");
    EXPECT_EQ(files[1].Name, "s2");

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::Batch(
                NSolomon::TKikimrKvBatchRequest(TabletId)
                    .ListFiles()
                    .ReadFile("s1")
                    .ReadFile("s2")));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TBatchResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        auto result = response->Get()->Value();
        ASSERT_EQ(result.ListFileQueryResults.size(), 1u);
        ASSERT_EQ(result.ListFileQueryResults[0].size(), 2u);
        EXPECT_EQ(result.ListFileQueryResults[0][0].Name, "s1");
        EXPECT_EQ(result.ListFileQueryResults[0][1].Name, "s2");
        ASSERT_EQ(result.ReadFileQueryResults.size(), 2u);
        EXPECT_EQ(result.ReadFileQueryResults[0], "snapshot 1");
        EXPECT_EQ(result.ReadFileQueryResults[1], "snapshot 2");
    }
}

TEST_F(KvClientActorFileTest, TabletOperationsDeadline) {
    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ListFiles(TabletId, Now() + TDuration::Seconds(10)));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Success());
        auto files = response->Get()->Value();
        ASSERT_EQ(files.size(), 0u);
    }

    {
        Runtime->Send(
            ClientActor,
            ListenerActor,
            NSolomon::NKv::TEvents::ListFiles(TabletId, Now() - TDuration::Seconds(10)));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(ListenerActor);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_EQ(error.AsKvError(), NKikimr::NMsgBusProxy::MSTATUS_TIMEOUT);
    }
}

TEST_F(KvClientActorTest, Retries) {
    auto Runtime = NSolomon::TTestActorRuntime::CreateInited();

    auto retryOpts = NSolomon::NKv::TRetryOpts{
        .MaxRetries = 6,
        .BackoffTime = TDuration::Seconds(1),
        .BackoffTimeMax = TDuration::Seconds(8),
        .BackoffFactor = 2,
    };
    auto client = Runtime->Register(NSolomon::NKv::CreateKvClientActor(Endpoint, Rpc.get(), retryOpts).Release());
    auto listener = Runtime->AllocateEdgeActor();

    Rpc->Reset();
    Rpc->SetHostDown();

    {
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::LocalTablets(/* retry = */ false));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TLocalTabletsResponse>(listener);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_TRUE(error.IsTransportError());
        EXPECT_EQ(error.AsTransportError(), grpc::UNAVAILABLE);
        EXPECT_EQ(Rpc->NumRequests(), 1u);
    }

    {
        auto time = Runtime->GetCurrentTime();
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::LocalTablets(/* retry = */ true));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TLocalTabletsResponse>(listener);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_TRUE(error.IsTransportError());
        EXPECT_EQ(error.AsTransportError(), grpc::UNAVAILABLE);
        EXPECT_EQ(Rpc->NumRequests(), 8u);
        EXPECT_EQ((Runtime->GetCurrentTime() - time).Seconds(), 31u);
    }

    {
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::LocalTablets(/* retry = */ true));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TLocalTabletsResponse>(listener);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_TRUE(error.IsTransportError());
        EXPECT_EQ(error.AsTransportError(), grpc::UNAVAILABLE);
        EXPECT_EQ(Rpc->NumRequests(), 15u);
    }

    Rpc->SetHostUp();

    Runtime->AdvanceCurrentTime(TDuration::Seconds(5));

    {
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::LocalTablets(/* retry = */ true));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TLocalTabletsResponse>(listener);
        ASSERT_TRUE(response->Get()->Success());
        EXPECT_EQ(Rpc->NumRequests(), 16u);
    }

    Rpc->SetHostDown(0, 3);

    {
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::LocalTablets(/* retry = */ true));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TLocalTabletsResponse>(listener);
        ASSERT_TRUE(response->Get()->Success());
        EXPECT_EQ(Rpc->NumRequests(), 20u);
    }
}

TEST_F(KvClientActorTest, KvRetries) {
    auto Runtime = NSolomon::TTestActorRuntime::CreateInited();

    auto retryOpts = NSolomon::NKv::TRetryOpts{
        .MaxRetries = 6,
        .BackoffTime = TDuration::Seconds(1),
        .BackoffTimeMax = TDuration::Seconds(8),
        .BackoffFactor = 2,
    };
    auto client = Runtime->Register(NSolomon::NKv::CreateKvClientActor(Endpoint, Rpc.get(), retryOpts).Release());
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::ListFiles(
                TabletId,
                /* deadline = */ TInstant::Now() - TDuration::Seconds(10),
                /* retry = */ false));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(listener);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_TRUE(error.IsKvError());
        EXPECT_EQ(error.AsKvError(), NKikimr::NMsgBusProxy::MSTATUS_TIMEOUT);
        EXPECT_EQ(Rpc->NumRequests(), 1u);
    }

    {
        auto time = Runtime->GetCurrentTime();
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::ListFiles(
                TabletId,
                /* deadline = */ TInstant::Now() - TDuration::Seconds(10),
                /* retry = */ true));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(listener);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_TRUE(error.IsKvError());
        EXPECT_EQ(error.AsKvError(), NKikimr::NMsgBusProxy::MSTATUS_TIMEOUT);
        EXPECT_EQ(Rpc->NumRequests(), 8u);
        EXPECT_EQ((Runtime->GetCurrentTime() - time).Seconds(), 31u);
    }

    {
        auto time = Runtime->GetCurrentTime();
        Runtime->Send(
            client,
            listener,
            NSolomon::NKv::TEvents::ListFiles(
                TabletId,
                /* deadline = */ TInstant::Now() - TDuration::Seconds(10),
                /* retry = */ true));
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(listener);
        ASSERT_TRUE(response->Get()->Fail());
        auto error = response->Get()->Error();
        ASSERT_TRUE(error.IsKvError());
        EXPECT_EQ(error.AsKvError(), NKikimr::NMsgBusProxy::MSTATUS_TIMEOUT);
        EXPECT_EQ(Rpc->NumRequests(), 15u);
        EXPECT_EQ((Runtime->GetCurrentTime() - time).Seconds(), 31u);
    }
}

TEST_F(KvClientActorFileTest, Cookies) {
    Runtime->Send(
        ClientActor,
        ListenerActor,
        NSolomon::NKv::TEvents::ListFiles(TabletId),
        /* flags = */ 0,
        /* cookie = */ 42);
    Runtime->Send(
        ClientActor,
        ListenerActor,
        NSolomon::NKv::TEvents::WriteFile(TabletId, "a", "content"),
        /* flags = */ 0,
        /* cookie = */ 10);
    {
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TListFilesResponse>(ListenerActor);
        ASSERT_EQ(response->Cookie, 42u);
    }
    {
        auto response = Runtime->GrabEdgeEvent<NSolomon::NKv::TEvents::TWriteFileResponse>(ListenerActor);
        ASSERT_EQ(response->Cookie, 10u);
    }
}
