#include "kv_client.h"

#include <util/generic/function.h>

using NSolomon::NKikimr::IKikimrRpc;
using NSolomon::NKikimr::TKikimrAsyncResponse;

namespace NSolomon {
namespace {

constexpr ui64 DOMAIN_UID = 1;

template <typename TApplyFn, typename TRes = TFunctionResult<TApplyFn>>
TKvResult<TRes> CheckStatusAndGetResultSync(TKikimrAsyncResponse future, TApplyFn&& fn) {
    using ::NKikimr::NMsgBusProxy::EResponseStatus;

    auto respOrErr = future.ExtractValueSync();
    if (respOrErr.Success()) {
        auto resp = respOrErr.Extract();
        auto status = static_cast<EResponseStatus>(resp.GetStatus());
        if (status == EResponseStatus::MSTATUS_OK) {
            try {
                if constexpr (std::is_same_v<TRes, void>) {
                    fn(std::move(resp));
                    return TKvResult<TRes>::FromValue();
                } else {
                    return TKvResult<TRes>::FromValue(fn(std::move(resp)));
                }
            } catch (...) {
                return TKvResult<TRes>::FromError(status, CurrentExceptionMessage());
            }
        }
        return TKvResult<TRes>::FromError(status, resp.GetErrorReason());
    }

    const auto& err = respOrErr.Error();
    auto status = static_cast<grpc::StatusCode>(err.GRpcStatusCode);
    return TKvResult<TRes>::FromError(status, err.Msg);
}

template <typename TApplyFn, typename TRes = TFunctionResult<TApplyFn>>
TAsyncKvResult<TRes> CheckStatusAndApply(const TKikimrAsyncResponse& future, TApplyFn&& fn) {
    return future.Apply([fn = std::forward<TApplyFn>(fn)](const TKikimrAsyncResponse& f) mutable {
        return CheckStatusAndGetResultSync<TApplyFn>(f, std::forward<TApplyFn>(fn));
    });
}

TAsyncKvResult<void> WaitAndCheckResponseStatus(IKikimrRpc* rpc, const TKikimrAsyncResponse& future) {
    using ::NKikimr::NMsgBusProxy::EResponseStatus;

    struct TCallback {
        IKikimrRpc* Rpc;
        NThreading::TPromise<TKvResult<void>> Promise;

        void operator()(const TKikimrAsyncResponse& future) {
            try {
                const auto& respOrErr = future.GetValueSync();
                if (respOrErr.Success()) {
                    const auto& resp = respOrErr.Value();
                    auto status = static_cast<EResponseStatus>(resp.GetStatus());
                    if (status == EResponseStatus::MSTATUS_OK) {
                        Promise.SetValue(TKvResult<void>::FromValue());
                    } else if (status == EResponseStatus::MSTATUS_INPROGRESS) {
                        NKikimrClient::TSchemeOperationStatus request;
                        *request.MutableFlatTxId() = resp.GetFlatTxId();
                        request.MutablePollOptions()->SetTimeout(10000); // 10 seconds
                        Rpc->SchemeOperationStatus(request).Subscribe(*this);
                    } else {
                        Promise.SetValue(TKvResult<void>::FromError(status, resp.GetErrorReason()));
                    }
                } else {
                    const auto& err = respOrErr.Error();
                    auto status = static_cast<grpc::StatusCode>(err.GRpcStatusCode);
                    Promise.SetValue(TKvResult<void>::FromError(status, err.Msg));
                }
            } catch (...) {
                // must never happen but in any case transfer an exception to a caller
                Promise.SetException(std::current_exception());
            }
        }
    };

    // Note: we want to keep size of this struct low so to avoid additional allocations in std::future.
    static_assert(sizeof(TCallback) <= 3 * sizeof(void*));
    // We also want this struct to be nothrow-copy-constructible, but this is not possible at the moment.
    static_assert(std::is_nothrow_copy_constructible_v<TCallback>);

    auto promise = NThreading::NewPromise<TKvResult<void>>();
    future.Subscribe(TCallback{rpc, promise});
    return promise.GetFuture();
}

NKikimrClient::TKeyValueRequest MakeKvRequest(ui64 tabletId, TInstant deadline) {
    NKikimrClient::TKeyValueRequest request;
    request.SetTabletId(tabletId);
    if (deadline && deadline != TInstant::Max()) {
        request.SetDeadlineInstantMs(deadline.MilliSeconds());
    }
    return request;
}

void RangeFill(const TKikimrKvRange& range, NKikimrClient::TKeyValueRequest_TKeyRange* r) {
    r->SetFrom(range.Begin);
    r->SetIncludeFrom(range.IncludeBegin);
    r->SetTo(range.End);
    r->SetIncludeTo(range.IncludeEnd);
}

void RangePrefix(const TString& prefix, NKikimrClient::TKeyValueRequest_TKeyRange* r) {
    r->SetFrom(prefix);
    r->SetIncludeFrom(true);
    // TODO: It looks like an error. All strings starting with "{prefix}~" will be missed
    r->SetTo(prefix + "~");
    r->SetIncludeTo(false);
}

void RangeExact(const TString& value, NKikimrClient::TKeyValueRequest_TKeyRange* r) {
    r->SetFrom(value);
    r->SetIncludeFrom(true);
    r->SetTo(value);
    r->SetIncludeTo(true);
}

void RangeAll(NKikimrClient::TKeyValueRequest_TKeyRange* r) {
    r->SetFrom("");
    r->SetIncludeFrom(true);
    // TODO: It looks like an error. All strings lexicographically greater than "~" will be missed (e.g. "~a")
    r->SetTo("~");
    r->SetIncludeTo(true);
}

class TListFilesContext: public TMoveOnly  {
public:
    TListFilesContext(NSolomon::NKikimr::IKikimrRpc* rpc,
            ui64 tabletId,
            TInstant deadline,
            const TString& prefix)
        : Rpc_(rpc)
        , TabletId_(tabletId)
        , Deadline_(deadline)
        , Range_(prefix, !prefix.empty(), prefix + "~", true)
    {
    }

    TListFilesContext(NSolomon::NKikimr::IKikimrRpc* rpc,
            ui64 tabletId,
            TInstant deadline,
            TKikimrKvRange&& readRange)
        : Rpc_(rpc)
        , TabletId_(tabletId)
        , Deadline_(deadline)
        , Range_(readRange)
    {
    }

    NSolomon::NKikimr::IKikimrRpc* GetRpc() const {
        return Rpc_;
    }

    void SetLimitBytes(ui64 limitBytes) {
        LimitBytes_ = limitBytes;
    }

    void ProhibitOverrun() {
        AllowOverrun_ = false;
    }

    NKikimrClient::TKeyValueRequest CreateRequest() const {
        auto request = MakeKvRequest(TabletId_, Deadline_);
        AddReadRangeRequest(request);
        return request;
    }

    void AddReadRangeRequest(NKikimrClient::TKeyValueRequest& request) const {
        auto* cmd = request.AddCmdReadRange();
        cmd->SetIncludeData(false);
        if (LimitBytes_ != 0) {
            cmd->SetLimitBytes(LimitBytes_);
        }

        auto* range = cmd->MutableRange();
        const TString& from = Files_.empty() ? Range_.Begin : Files_.back().back().Name;
        range->SetFrom(from);
        range->SetIncludeFrom(Files_.empty() && Range_.IncludeBegin);
        range->SetTo(Range_.End);
        range->SetIncludeTo(Range_.IncludeEnd);
    }

    void AddReadRangeResult(::NKikimrClient::TKeyValueResponse_TReadRangeResult&& readRangeResult) {
        Status_ = static_cast<NKikimrProto::EReplyStatus>(readRangeResult.GetStatus());
        if (!IsValidStatus()) {
            Error_ = TKvClientError(Status_, "Unexpected kikimr read range status");
            return;
        }

        TVector<TFileInfo> files;
        files.reserve(readRangeResult.PairSize());
        for (auto& pair: *readRangeResult.MutablePair()) {
            files.emplace_back(TFileInfo{
                    .Name = std::move(*pair.MutableKey()),
                    .SizeBytes = pair.GetValueSize(),
                    .CreatedAt = TInstant::Seconds(pair.GetCreationUnixTime()),
            });
        }
        if (!files.empty()) {
            Files_.push_back(std::move(files));
        }
    }

    bool HasMoreData() const {
        return AllowOverrun_ && Status_ == NKikimrProto::EReplyStatus::OVERRUN;
    };

    void SetError(TKvClientError&& error) {
        Error_ = error;
    }

    const TKvClientError& GetError() const {
        return Error_.value();
    }

    bool Succeeded() const {
        return !Error_.has_value();
    }

    [[nodiscard]] TVector<TFileInfo> ExtractFiles() {
        if (Files_.empty()) {
            return TVector<TFileInfo>{};
        }

        if (Files_.size() > 1) {
            size_t totalSize = 0;
            for (const auto& f: Files_) {
                totalSize += f.size();
            }
            Files_[0].reserve(totalSize);
            for (size_t i = 1; i < Files_.size(); i++) {
                std::move(Files_[i].begin(), Files_[i].end(), std::back_inserter(Files_[0]));
            }
        }
        return std::move(Files_[0]);
    }

private:
    bool IsValidStatus() const {
        using namespace NKikimrProto;
        switch (Status_) {
            case OK:
            case NODATA:
            case OVERRUN:
                return true;
            default:
                return false;
        };
    }

private:
    NSolomon::NKikimr::IKikimrRpc * const Rpc_;
    const ui64 TabletId_;
    const TInstant Deadline_;
    const TKikimrKvRange Range_;
    NKikimrProto::EReplyStatus Status_{NKikimrProto::EReplyStatus::UNKNOWN};
    TVector<TVector<TFileInfo>> Files_;
    std::optional<TKvClientError> Error_;
    ui64 LimitBytes_{0};
    bool AllowOverrun_{true};
};

class TBatchContext : public TMoveOnly {
public:
    using TReadRangeResults = ::google::protobuf::RepeatedPtrField<::NKikimrClient::TKeyValueResponse_TReadRangeResult>;
    using TReadResults = ::google::protobuf::RepeatedPtrField<::NKikimrClient::TKeyValueResponse_TReadResult>;

    TBatchContext(NSolomon::NKikimr::IKikimrRpc* rpc, const NKikimrClient::TKeyValueRequest& batchRequest)
        : Rpc_(rpc)
        , TabletId_(batchRequest.GetTabletId())
        , Deadline_(TInstant::Seconds(batchRequest.GetDeadlineInstantMs()))
    {
        for (ui32 i = 0; i < static_cast<ui32>(batchRequest.GetCmdReadRange().size()); i++) {
            const auto& r = batchRequest.GetCmdReadRange(i).GetRange();
            ListFileContexts_.push_back(std::make_shared<TListFilesContext>(
                    Rpc_,
                    TabletId_,
                    Deadline_,
                    TKikimrKvRange(r.GetFrom(), r.GetIncludeFrom(), r.GetTo(), r.GetIncludeTo())));
            UnfinishedContextIds_[i] = i;
        }
    }

    NSolomon::NKikimr::IKikimrRpc* GetRpc() const {
        return Rpc_;
    }

    void AddReadRangeResults(TReadRangeResults&& results) {
        if (!Succeeded()) {
            return;
        }

        for (i32 i = 0; i < results.size(); i++) {
            auto ctxId = UnfinishedContextIds_[i];
            const auto& ctx = ListFileContexts_[ctxId];
            ctx->AddReadRangeResult(std::move(results[i]));
            if (!ctx->Succeeded()) {
                Error_ = ctx->GetError();
                return;
            }
        }
    }

    const TKvClientError& GetError() const {
        return Error_.value();
    }

    bool Succeeded() const {
        return !Error_.has_value();
    }

    void AddReadResults(const TReadResults& readResults) {
        if (!Succeeded()) {
            return;
        }

        for (const auto& readResult: readResults) {
            ReadFileResults_.emplace_back(readResult.GetValue());
        }
    }

    bool HasMoreData() const {
        if (!Succeeded()) {
            return false;
        }

        for (const auto& ctx: ListFileContexts_) {
            if (ctx->HasMoreData()) {
                return true;
            }
        }
        return false;
    }

    NKikimrClient::TKeyValueRequest CreateRequest() {
        auto batchRequest = MakeKvRequest(TabletId_, Deadline_);
        UnfinishedContextIds_.clear();
        for (ui32 i = 0; i < ListFileContexts_.size(); i++) {
            if (ListFileContexts_[i]->HasMoreData()) {
                const auto index = static_cast<ui32>(UnfinishedContextIds_.size());
                UnfinishedContextIds_[index] = i;
                ListFileContexts_[i]->AddReadRangeRequest(batchRequest);
            }
        }
        return batchRequest;
    }

    [[nodiscard]] TKikimrKvBatchResult ExtractResult() {
        TKikimrKvBatchResult result;
        for (const auto& ctx: ListFileContexts_) {
            result.ListFileQueryResults.push_back(std::move(ctx->ExtractFiles()));
        }
        result.ReadFileQueryResults = std::move(ReadFileResults_);
        return result;
    }

private:
    NSolomon::NKikimr::IKikimrRpc* const Rpc_;
    const ui64 TabletId_;
    const TInstant Deadline_;
    std::vector<std::shared_ptr<TListFilesContext>> ListFileContexts_;
    std::unordered_map<ui32, ui32> UnfinishedContextIds_;
    TVector<TString> ReadFileResults_;
    std::optional<TKvClientError> Error_;
};

} // namespace

TKikimrKvBatchRequest::TKikimrKvBatchRequest(ui64 tabletId, TInstant deadline)
    : Request_{MakeKvRequest(tabletId, deadline)}
{
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::ListFiles()& {
    auto* cmdReadRange = Request_.AddCmdReadRange();
    cmdReadRange->SetIncludeData(false);
    RangeAll(cmdReadRange->MutableRange());

    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::ListFiles()&& {
    return std::move(ListFiles());
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::ListPrefix(const TString& prefix)& {
    auto* cmdReadRange = Request_.AddCmdReadRange();
    cmdReadRange->SetIncludeData(false);
    RangePrefix(prefix, cmdReadRange->MutableRange());
    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::ListPrefix(const TString& prefix)&& {
    return std::move(ListPrefix(prefix));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::ReadFile(const TString& name, ui64 offset, ui64 size)& {
    auto* cmdRead = Request_.AddCmdRead();
    cmdRead->SetKey(name);
    if (offset) {
        cmdRead->SetOffset(offset);
    }
    if (size) {
        cmdRead->SetSize(size);
    }

    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::ReadFile(const TString& name, ui64 offset, ui64 size)&& {
    return std::move(ReadFile(name, offset, size));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::WriteFile(const TString& name, const TString& data)& {
    auto* cmdWrite = Request_.AddCmdWrite();
    cmdWrite->SetKey(name);
    cmdWrite->SetValue(data);

    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::WriteFile(const TString& name, const TString& data)&& {
    return std::move(WriteFile(name, data));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::RenameFile(const TString& source, const TString& target)& {
    auto* cmdRename = Request_.AddCmdRename();
    cmdRename->SetOldKey(source);
    cmdRename->SetNewKey(target);

    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::RenameFile(const TString& source, const TString& target)&& {
    return std::move(RenameFile(source, target));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::CopyFile(const TString& from, const TString& to)& {
    return CopyFiles(TKikimrKvRange{from, true, from, true}, from, to);
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::CopyFile(const TString& from, const TString& to)&& {
    return std::move(CopyFile(from, to));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::CopyFiles(const TKikimrKvRange& from, const TString& prefixToRemove, const TString& prefixToAdd)& {
    auto* cmdCopy = Request_.AddCmdCopyRange();
    cmdCopy->SetPrefixToRemove(prefixToRemove);
    cmdCopy->SetPrefixToAdd(prefixToAdd);
    RangeFill(from, cmdCopy->MutableRange());
    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::CopyFiles(const TKikimrKvRange& from, const TString& prefixToRemove, const TString& prefixToAdd)&& {
    return std::move(CopyFiles(from, prefixToRemove, prefixToAdd));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::RemoveFile(const TString& file)& {
    return RemoveFiles(TKikimrKvRange{file, true, file, true});
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::RemoveFile(const TString& file)&& {
    return std::move(RemoveFile(file));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::RemoveFiles(const TKikimrKvRange& file)& {
    auto* cmdDelete = Request_.AddCmdDeleteRange();
    RangeFill(file, cmdDelete->MutableRange());
    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::RemoveFiles(const TKikimrKvRange& file)&& {
    return std::move(RemoveFiles(file));
}

TKikimrKvBatchRequest& TKikimrKvBatchRequest::RemovePrefix(const TString& prefix)& {
    auto* cmdCopy = Request_.AddCmdCopyRange();
    cmdCopy->SetPrefixToRemove(prefix);
    RangePrefix(prefix, cmdCopy->MutableRange());

    auto* cmdDelete = Request_.AddCmdDeleteRange();
    RangePrefix(prefix, cmdDelete->MutableRange());
    return *this;
}

TKikimrKvBatchRequest TKikimrKvBatchRequest::RemovePrefix(const TString& prefix)&& {
    return std::move(RemovePrefix(prefix));
}

TAsyncKvResult<void> TKikimrKvClient::CreateSolomonVolume(
        const TString& path,
        ui64 partitionCount,
        ui32 channelProfileId)
{
    TStringBuf dir, name;
    TStringBuf(path).RSplit('/', dir, name);

    NKikimrClient::TSchemeOperation request;
    if (auto* modifyScheme = request.MutableTransaction()->MutableModifyScheme()) {
        modifyScheme->SetOperationType(NKikimrSchemeOp::EOperationType::ESchemeOpCreateSolomonVolume);
        modifyScheme->SetWorkingDir(TString(dir));

        if (auto* createSolomonVolume = modifyScheme->MutableCreateSolomonVolume()) {
            createSolomonVolume->SetName(TString(name));
            createSolomonVolume->SetPartitionCount(partitionCount);
            createSolomonVolume->SetChannelProfileId(channelProfileId);
        }
    }

    auto future = Rpc_->SchemeOperation(request);
    return WaitAndCheckResponseStatus(Rpc_, future);
}

TAsyncKvResult<void> TKikimrKvClient::DropSolomonVolume(const TString& path) {
    TStringBuf dir, name;
    TStringBuf(path).RSplit('/', dir, name);

    NKikimrClient::TSchemeOperation request;
    if (auto* modifyScheme = request.MutableTransaction()->MutableModifyScheme()) {
        modifyScheme->SetOperationType(NKikimrSchemeOp::EOperationType::ESchemeOpDropSolomonVolume);
        modifyScheme->SetWorkingDir(TString(dir));
        modifyScheme->MutableDrop()->SetName(TString(name));
    }

    auto future = Rpc_->SchemeOperation(request);
    return WaitAndCheckResponseStatus(Rpc_, future);
}

TAsyncKvResult<TVector<ui64>> TKikimrKvClient::ResolveTablets(ui64 ownerId, const TVector<ui64>& ownerIdxs) {
    NKikimrClient::THiveCreateTablet request;
    request.SetDomainUid(DOMAIN_UID);
    for (ui64 ownerIdx: ownerIdxs) {
        auto* lookup = request.AddCmdLookupTablet();
        lookup->SetOwnerId(ownerId);
        lookup->SetOwnerIdx(ownerIdx);
    }

    auto future = Rpc_->HiveCreateTablet(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse& r) {
        if (r.LookupTabletResultSize() > 0) {
            ui32 status = r.GetLookupTabletResult(0).GetStatus();
            ui64 tabletId = r.GetLookupTabletResult(0).GetTabletId();
            if (status == NKikimrProto::EReplyStatus::NODATA && tabletId == 0) {
                // actually hive respond that there is no such ownerId in his internal table
                return TVector<ui64>{};
            }
        }

        TVector<ui64> tabletIds(::Reserve(r.LookupTabletResultSize()));
        for (size_t i = 0; i < r.LookupTabletResultSize(); i++) {
            tabletIds.push_back(r.GetLookupTabletResult(i).GetTabletId());
        }
        return tabletIds;
    });
}

TAsyncKvResult<TVector<ui64>> TKikimrKvClient::ResolveTablets(const TString& solomonVolumePath) {
    NKikimrClient::TSchemeDescribe request;
    request.SetPath(solomonVolumePath);

    auto future = Rpc_->SchemeDescribe(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse& r) {
        const auto& pathDescr = r.GetPathDescription();
        const auto& solomonDescr = pathDescr.GetSolomonDescription();

        TVector<ui64> tabletIds(::Reserve(solomonDescr.PartitionsSize()));
        for (size_t i = 0; i < solomonDescr.PartitionsSize(); i++) {
            tabletIds.push_back(solomonDescr.GetPartitions(i).GetTabletId());
        }
        return tabletIds;
    });
}

TAsyncKvResult<TVector<ui64>> TKikimrKvClient::LocalTablets() {
    NKikimrClient::TLocalEnumerateTablets request;
    request.SetDomainUid(DOMAIN_UID);
    request.SetTabletType(NKikimrTabletBase::TTabletTypes_EType_KeyValue);

    auto future = Rpc_->LocalEnumerateTablets(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse& r) {
        TVector<ui64> tabletIds(::Reserve(r.TabletInfoSize()));
        for (size_t i = 0; i < r.TabletInfoSize(); i++) {
            tabletIds.push_back(r.GetTabletInfo(i).GetTabletId());
        }
        return tabletIds;
    });
}

TAsyncKvResult<TVector<TTabletInfo>> TKikimrKvClient::TabletsInfo(const TVector<ui64>& tabletIds) {
    NKikimrClient::TTabletStateRequest request;
    request.SetTabletType(NKikimrTabletBase::TTabletTypes_EType_KeyValue);
    request.SetAlive(true);
    for (ui64 tabletId: tabletIds) {
        request.AddTabletIDs(tabletId);
    }

    auto future = Rpc_->TabletStateRequest(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse& r) {
        TVector<TTabletInfo> result(::Reserve(r.TabletStateInfoSize()));
        for (size_t i = 0; i < r.TabletStateInfoSize(); i++) {
            const auto& tabletState = r.GetTabletStateInfo(i);
            result.push_back(TTabletInfo{
                tabletState.GetTabletId(),
                TInstant::MilliSeconds(tabletState.GetChangeTime()),
                tabletState.GetHost()
            });
        }

        return result;
    });
}

TAsyncKvResult<ui64> TKikimrKvClient::IncrementGeneration(ui64 tabletId, TInstant deadline) {
    auto request = MakeKvRequest(tabletId, deadline);
    request.MutableCmdIncrementGeneration();

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse& r) {
        const auto& result = r.GetIncrementGenerationResult();
        return result.GetGeneration();
    });
}

TAsyncKvResult<TVector<TFileInfo>> TKikimrKvClient::ListFiles(ui64 tabletId, TInstant deadline) {
    return ListPrefix(tabletId, "", deadline);
}

TAsyncKvResult<TVector<TFileInfo>> TKikimrKvClient::ListPrefix(
        ui64 tabletId,
        const TString& prefix,
        TInstant deadline)
{
    struct TCallback {
        std::shared_ptr<TListFilesContext> Context;
        NThreading::TPromise<TKvResult<TVector<TFileInfo>>> Promise;

        void operator()(const TKikimrAsyncResponse& future) {
            try {
                // lambda is guaranteed to be executed synchronously, so capture of this is safe
                auto res = CheckStatusAndGetResultSync(future, [this](NKikimrClient::TResponse&& r) {
                    Context->AddReadRangeResult(std::move(*r.MutableReadRangeResult()->Mutable(0)));
                });

                if (res.Fail()) {
                    Promise.SetValue(TKvResult<TVector<TFileInfo>>::FromError(res.ExtractError()));
                    return;
                }
                if (!Context->Succeeded()) {
                    Promise.SetValue(TKvResult<TVector<TFileInfo>>::FromError(Context->GetError()));
                    return;
                }
                if (!Context->HasMoreData()) {
                    Promise.SetValue(TKvResult<TVector<TFileInfo>>::FromValue(
                            std::move(Context->ExtractFiles())));
                    return;
                }

                auto request = Context->CreateRequest();
                auto nextChunkFuture = Context->GetRpc()->KeyValue(request);
                nextChunkFuture.Subscribe(*this);
            } catch (...) {
                // must never happen but in any case transfer an exception to a caller
                Promise.SetException(std::current_exception());
            }
        }
    };

    TCallback callback{
        .Context = std::make_shared<TListFilesContext>(Rpc_, tabletId, deadline, prefix),
        .Promise = NThreading::NewPromise<TKvResult<TVector<TFileInfo>>>()
    };

    if (Options_) {
        callback.Context->SetLimitBytes(Options_->LimitBytes);
        if (!Options_->AllowOverrun) {
            callback.Context->ProhibitOverrun();
        }
    }

    auto request = callback.Context->CreateRequest();
    auto result = callback.Promise.GetFuture();
    Rpc_->KeyValue(request).Subscribe(std::move(callback));
    return result;
}

TAsyncKvResult<TString> TKikimrKvClient::ReadFile(
        ui64 tabletId,
        const TString& name,
        ui64 offset, ui64 size,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdRead()) {
        cmd->SetKey(name);
        if (offset) {
            cmd->SetOffset(offset);
        }
        if (size) {
            cmd->SetSize(size);
        }
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse& r) {
        for (auto& readResult: r.GetReadResult()) {
            return readResult.GetValue();
        }
        return TString{};
    });
}

TAsyncKvResult<void> TKikimrKvClient::WriteFile(
        ui64 tabletId,
        const TString& name,
        const TString& data,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdWrite()) {
        cmd->SetKey(name);
        cmd->SetValue(data);
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::ConcatFiles(
        ui64 tabletId,
        const TVector<TString>& inputs,
        const TString& output,
        bool keepInputs,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdConcat()) {
        for (const auto &input: inputs) {
            cmd->AddInputKeys(input);
        }
        cmd->SetOutputKey(output);
        cmd->SetKeepInputs(keepInputs);
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::RenameFile(
        ui64 tabletId,
        const TString& source,
        const TString& target,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdRename()) {
        cmd->SetOldKey(source);
        cmd->SetNewKey(target);
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::CopyFile(
        ui64 tabletId,
        const TString& from,
        const TString& to,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdCopyRange()) {
        cmd->SetPrefixToRemove(from);
        cmd->SetPrefixToAdd(to);
        RangeExact(from, cmd->MutableRange());
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::CopyFiles(
        ui64 tabletId,
        const TKikimrKvRange& from,
        const TString& prefixToRemove,
        const TString& prefixToAdd,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdCopyRange()) {
        cmd->SetPrefixToRemove(prefixToRemove);
        cmd->SetPrefixToAdd(prefixToAdd);
        RangeFill(from, cmd->MutableRange());
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::RemoveFile(
        ui64 tabletId,
        const TString& file,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdDeleteRange()) {
        RangeExact(file, cmd->MutableRange());
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::RemoveFiles(
        ui64 tabletId,
        const TKikimrKvRange& range,
        TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdDeleteRange()) {
        RangeFill(range, cmd->MutableRange());
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<void> TKikimrKvClient::RemovePrefix(
        ui64 tabletId, const TString& prefix, TInstant deadline)
{
    auto request = MakeKvRequest(tabletId, deadline);
    if (auto* cmd = request.AddCmdCopyRange()) {
        cmd->SetPrefixToRemove(prefix);
        RangePrefix(prefix, cmd->MutableRange());
    }
    if (auto* cmd = request.AddCmdDeleteRange()) {
        RangePrefix(prefix, cmd->MutableRange());
    }

    auto future = Rpc_->KeyValue(request);
    return CheckStatusAndApply(future, [](const NKikimrClient::TResponse&) {});
}

TAsyncKvResult<TKikimrKvBatchResult> TKikimrKvClient::BatchRequest(const TKikimrKvBatchRequest& req) {
    struct TCallback {
        std::shared_ptr<TBatchContext> BatchContext;
        NThreading::TPromise<TKvResult<TKikimrKvBatchResult>> Promise;

        void operator()(const TKikimrAsyncResponse& future) {
            try {
                // lambda is guaranteed to be executed synchronously, so capture of this is safe
                auto res = CheckStatusAndGetResultSync(future, [this](NKikimrClient::TResponse&& r) {
                    BatchContext->AddReadRangeResults(std::move(*r.MutableReadRangeResult()));
                    BatchContext->AddReadResults(r.GetReadResult());
                });

                if (res.Fail()) {
                    Promise.SetValue(TKvResult<TKikimrKvBatchResult>::FromError(res.ExtractError()));
                    return;
                }
                if (!BatchContext->Succeeded()) {
                    Promise.SetValue(TKvResult<TKikimrKvBatchResult>::FromError(BatchContext->GetError()));
                    return;
                }
                if (!BatchContext->HasMoreData()) {
                    Promise.SetValue(TKvResult<TKikimrKvBatchResult>::FromValue(
                            std::move(BatchContext->ExtractResult())));
                    return;
                }

                auto request = BatchContext->CreateRequest();
                auto nextChunkFuture = BatchContext->GetRpc()->KeyValue(request);
                nextChunkFuture.Subscribe(*this);
            } catch (...) {
                // must never happen but in any case transfer an exception to a caller
                Promise.SetException(std::current_exception());
            }
        }
    };

    TCallback callback{
        .BatchContext = std::make_shared<TBatchContext>(Rpc_, req.Request_),
        .Promise = NThreading::NewPromise<TKvResult<TKikimrKvBatchResult>>()
    };

    auto result = callback.Promise.GetFuture();
    Rpc_->KeyValue(req.Request_).Subscribe(std::move(callback));
    return result;
}

} // namespace NSolomon

template <>
void Out<NSolomon::TKikimrKvRange>(IOutputStream& out, const NSolomon::TKikimrKvRange& x) {
    out << (x.IncludeBegin ? '[' : '(') << x.Begin
        << ".."
        << x.End << (x.IncludeEnd ? ']' : ')');
}
