#pragma once

#include "error.h"

#include <solomon/libs/cpp/clients/kikimr/rpc.h>

#include <library/cpp/grpc/client/grpc_common.h>
#include <library/cpp/threading/future/future.h>

namespace NSolomon {

template <typename T>
using TKvResult = TErrorOr<T, TKvClientError>;

template <typename T>
using TAsyncKvResult = NThreading::TFuture<TKvResult<T>>;

///////////////////////////////////////////////////////////////////////////////
// KV client
///////////////////////////////////////////////////////////////////////////////

struct TTabletInfo {
    ui64 TabletId;
    TInstant ChangedAt;
    TString Host;
};

struct TFileInfo {
    TString Name;
    ui32 SizeBytes;
    TInstant CreatedAt;
};

/// Result of execution of a single batch request.
struct TKikimrKvBatchResult {
    /// All results for `ListFiles` command, in order as `ListFiles` were added to the batch.
    TVector<TVector<TFileInfo>> ListFileQueryResults;

    /// All results for `ReadFile` command, in order as `ReadFile` were added to the batch.
    TVector<TString> ReadFileQueryResults;
};

class TKikimrKvClient;

/// Range of files.
struct TKikimrKvRange {
    TString Begin, End;
    bool IncludeBegin, IncludeEnd;

    TKikimrKvRange(TString begin, TString end)
        : TKikimrKvRange(std::move(begin), true, std::move(end), false)
    {
    }

    TKikimrKvRange(TString begin, bool includeBegin, TString end, bool includeEnd)
        : Begin(std::move(begin))
        , End(std::move(end))
        , IncludeBegin{includeBegin}
        , IncludeEnd{includeEnd}
    {
    }

    TKikimrKvRange WithIncludeBegin(bool includeBegin) && {
        IncludeBegin = includeBegin;
        return std::move(*this);
    }

    TKikimrKvRange WithIncludeEnd(bool includeEnd) && {
        IncludeEnd = includeEnd;
        return std::move(*this);
    }
};

/// A request to observe and modify data in a KV tablet.
///
/// This request can contain several commands. All commands from a single request are executed atomically,
/// i.e. in single transaction.
///
/// Commands of the same type are executed in order as they were added to the batch. If, for example, you call
/// `batch.ReadFile('/a')` and then `batch.ReadFile('/b')`, file `/a` will be read before file `/b`, and contents
/// of file `/a` will appear before contents of file `/b` in the response.
///
/// Commands of different types are executed in unspecified order. For example, tablet may decide to execute
/// all `ListFiles` first and then all `ReadFile`, or it may do it the other way around. In practice, there is
/// some defined ordering, but it is not a part of the public interface.
class TKikimrKvBatchRequest {
    friend class TKikimrKvClient;

public:
    explicit TKikimrKvBatchRequest(ui64 tabletId, TInstant deadline = TDuration::Seconds(60).ToDeadLine());

public:
    TKikimrKvBatchRequest ListFiles()&&;
    TKikimrKvBatchRequest& ListFiles()&;

    TKikimrKvBatchRequest ListPrefix(const TString& prefix)&&;
    TKikimrKvBatchRequest ListPrefix(const TString& prefix)&;

    TKikimrKvBatchRequest ReadFile(const TString& name, ui64 offset = 0, ui64 size = 0)&&;
    TKikimrKvBatchRequest& ReadFile(const TString& name, ui64 offset = 0, ui64 size = 0)&;

    TKikimrKvBatchRequest WriteFile(const TString& name, const TString& data)&&;
    TKikimrKvBatchRequest& WriteFile(const TString& name, const TString& data)&;

    TKikimrKvBatchRequest RenameFile(const TString& source, const TString& target)&&;
    TKikimrKvBatchRequest& RenameFile(const TString& source, const TString& target)&;

    TKikimrKvBatchRequest CopyFile(const TString& from, const TString& to)&&;
    TKikimrKvBatchRequest& CopyFile(const TString& from, const TString& to)&;

    TKikimrKvBatchRequest CopyFiles(const TKikimrKvRange& from, const TString& prefixToRemove, const TString& prefixToAdd)&&;
    TKikimrKvBatchRequest& CopyFiles(const TKikimrKvRange& from, const TString& prefixToRemove, const TString& prefixToAdd)&;

    TKikimrKvBatchRequest RemoveFile(const TString& file)&&;
    TKikimrKvBatchRequest& RemoveFile(const TString& file)&;

    TKikimrKvBatchRequest RemoveFiles(const TKikimrKvRange& file)&&;
    TKikimrKvBatchRequest& RemoveFiles(const TKikimrKvRange& file)&;

    TKikimrKvBatchRequest RemovePrefix(const TString& prefix)&&;
    TKikimrKvBatchRequest& RemovePrefix(const TString& prefix)&;

private:
    NKikimrClient::TKeyValueRequest Request_;
};

struct TKikimrKvClientOptions {
    ui64 LimitBytes{0};
    bool AllowOverrun{true};
};

class TKikimrKvClient {
public:
    explicit TKikimrKvClient(NSolomon::NKikimr::IKikimrRpc* rpc) noexcept
        : Rpc_{rpc}
    {
    }

    // for testing
    explicit TKikimrKvClient(
            NSolomon::NKikimr::IKikimrRpc* rpc,
            std::unique_ptr<TKikimrKvClientOptions>&& options) noexcept
        : Rpc_{rpc}
        , Options_{std::move(options)}
    {
    }

    TAsyncKvResult<void> CreateSolomonVolume(const TString& path, ui64 partitionCount, ui32 channelProfileId = 0);

    TAsyncKvResult<void> DropSolomonVolume(const TString& path);

    TAsyncKvResult<TVector<ui64>> ResolveTablets(ui64 ownerId, const TVector<ui64>& ownerIdxs);

    TAsyncKvResult<TVector<ui64>> ResolveTablets(const TString& solomonVolumePath);

    TAsyncKvResult<TVector<ui64>> LocalTablets();

    TAsyncKvResult<TVector<TTabletInfo>> TabletsInfo(const TVector<ui64>& tabletIds);

    TAsyncKvResult<ui64> IncrementGeneration(
            ui64 tabletId,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<TVector<TFileInfo>> ListFiles(
            ui64 tabletId,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<TVector<TFileInfo>> ListPrefix(
            ui64 tabletId, const TString& prefix,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<TString> ReadFile(
            ui64 tabletId, const TString& name, ui64 offset = 0, ui64 size = 0,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> WriteFile(
            ui64 tabletId, const TString& name, const TString& data,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> ConcatFiles(
            ui64 tabletId, const TVector<TString>& inputs, const TString& output, bool keepInputs = false,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> RenameFile(
            ui64 tabletId, const TString& source, const TString& target,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> CopyFile(
            ui64 tabletId, const TString& from, const TString& to,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> CopyFiles(
            ui64 tabletId, const TKikimrKvRange& from, const TString& prefixToRemove, const TString& prefixToAdd,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> RemoveFile(
            ui64 tabletId, const TString& file,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> RemoveFiles(
            ui64 tabletId, const TKikimrKvRange& range,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<void> RemovePrefix(
            ui64 tabletId, const TString& prefix,
            TInstant deadline = TDuration::Seconds(60).ToDeadLine());

    TAsyncKvResult<TKikimrKvBatchResult> BatchRequest(const TKikimrKvBatchRequest& req);

private:
    NSolomon::NKikimr::IKikimrRpc* Rpc_;
    std::unique_ptr<TKikimrKvClientOptions> Options_;
};

} // namespace NSolomon
