#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/kv/kv_client.h>
#include <solomon/libs/cpp/kv/testlib/testlib.h>

using namespace NSolomon;

namespace {

constexpr size_t SeriesLen = 4;

class TFileNamesGenerator {
public:
    explicit TFileNamesGenerator(const TString& name)
        : FileNameTemplate_("%04u." + name + ".%012u")
        , NameLen_(name.length() + 2) // dot to the left and dot to the right
    {
    }

    void NextSeries() {
        SeriesNumber_++;
        FileNumber_ = 0;
    }

    TString NextFileName() {
        return Sprintf(FileNameTemplate_.c_str(), SeriesNumber_, FileNumber_++);
    }

    std::pair<i32, i32> GetSeriesAndFileNumber(TStringBuf fileName) {
        auto series = fileName.NextTokAt(SeriesLen);
        auto number = fileName.SplitOffAt(NameLen_);
        std::pair<i32, i32> res;
        if (!TryFromString(series, res.first)) {
            res.first = -1;
            return res;
        }
        if (!TryFromString(number, res.second)) {
            res.second = -1;
        }
        return res;
    }

private:
    const TString FileNameTemplate_;
    const size_t NameLen_;
    ui32 SeriesNumber_{0};
    ui32 FileNumber_{0};
};
}

class TKvClientTest: public NKv::TTestFixture {
};

TEST_F(TKvClientTest, CreateRpc) {
    auto rpc = CreateRpc();
    ASSERT_TRUE(rpc);
}

class TKvClientTabletTest: public NKv::TTestFixtureTablet<1> {
};

TEST_F(TKvClientTabletTest, CreateAndDropSolomonVolume) {
    auto solomonVolumePath = SolomonVolumePath + "_2";

    Client->CreateSolomonVolume(solomonVolumePath, 2).GetValueSync();

    auto tabletIds = Client->ResolveTablets(solomonVolumePath)
            .ExtractValueSync()
            .Extract();
    ASSERT_EQ(tabletIds.size(), 2u);

    auto localTabletIds = Client->LocalTablets()
            .ExtractValueSync()
            .Extract();
    for (auto tabletId: tabletIds) {
        EXPECT_TRUE(Find(localTabletIds, tabletId) != localTabletIds.end());
    }

    Client->DropSolomonVolume(solomonVolumePath).GetValueSync();

    auto status = Client->ResolveTablets(solomonVolumePath).ExtractValueSync();
    EXPECT_FALSE(status.Success());
    EXPECT_EQ("Path not found", status.Error().Message());
}

TEST_F(TKvClientTabletTest, CreateSolomonVolumeError) {
    auto status = Client->CreateSolomonVolume("_invalid_db_", 1).ExtractValueSync();
    EXPECT_FALSE(status.Success());
    EXPECT_EQ("Path does not exist", status.Error().Message());
}

TEST_F(TKvClientTabletTest, ListFiles) {
    {
        auto files = Client->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "").GetValueSync();
    StableClient->WriteFile(TabletId, "b", "").GetValueSync();

    {
        auto files = Client->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(files[1].Name, "b");
    }
}

TEST_F(TKvClientTabletTest, ReadWriteFile) {
    Client->WriteFile(TabletId, "a", "content").GetValueSync();
    EXPECT_EQ(Client->ReadFile(TabletId, "a").GetValueSync().Value(), "content");
    EXPECT_EQ(Client->ReadFile(TabletId, "b").GetValueSync().Value(), "");
}

TEST_F(TKvClientTabletTest, RenameFile) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();

    Client->RenameFile(TabletId, "a", "x").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[1].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[2].Name, "x");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x").GetValueSync().Value(), "a content");
    }
}

TEST_F(TKvClientTabletTest, CopyFile) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b", "b content").GetValueSync();
    StableClient->WriteFile(TabletId, "c", "c content").GetValueSync();

    Client->CopyFile(TabletId, "b", "d").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 4u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b").GetValueSync().Value(), "b content");
        EXPECT_EQ(files[2].Name, "c");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c").GetValueSync().Value(), "c content");
        EXPECT_EQ(files[3].Name, "d");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "d").GetValueSync().Value(), "b content");
    }
}

TEST_F(TKvClientTabletTest, CopyFiles) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b3", "b3 content").GetValueSync();
    StableClient->WriteFile(TabletId, "c", "c content").GetValueSync();

    Client->CopyFiles(TabletId, TKikimrKvRange{"b", "b~"}, "b", "x1").GetValueSync();
    Client->CopyFiles(TabletId, TKikimrKvRange{"b", false, "c", true}, "b", "x2").GetValueSync();
    Client->CopyFiles(TabletId, TKikimrKvRange{"b1", false, "c", true}, "b", "x3").GetValueSync();
    Client->CopyFiles(TabletId, TKikimrKvRange{"b", "b"}, "b", "x4").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 13u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "b1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[2].Name, "b2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[3].Name, "b3");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "b3").GetValueSync().Value(), "b3 content");
        EXPECT_EQ(files[4].Name, "c");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c").GetValueSync().Value(), "c content");

        EXPECT_EQ(files[5].Name, "x11");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x11").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[6].Name, "x12");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x12").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[7].Name, "x13");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x13").GetValueSync().Value(), "b3 content");

        EXPECT_EQ(files[8].Name, "x21");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x21").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[9].Name, "x22");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x22").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[10].Name, "x23");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x23").GetValueSync().Value(), "b3 content");

        EXPECT_EQ(files[11].Name, "x32");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x32").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[12].Name, "x33");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "x33").GetValueSync().Value(), "b3 content");
    }
}

TEST_F(TKvClientTabletTest, RemoveFile) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b", "b content").GetValueSync();
    StableClient->WriteFile(TabletId, "c", "c content").GetValueSync();

    Client->RemoveFile(TabletId, "b").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "c");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c").GetValueSync().Value(), "c content");
    }
}

TEST_F(TKvClientTabletTest, RemoveFiles) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b3", "b3 content").GetValueSync();
    StableClient->WriteFile(TabletId, "c", "c content").GetValueSync();

    Client->RemoveFiles(TabletId, TKikimrKvRange{"b", "b~"}).GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u);
        EXPECT_EQ(files[0].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
        EXPECT_EQ(files[1].Name, "c");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "c").GetValueSync().Value(), "c content");
    }
}

TEST_F(TKvClientTabletTest, RemovePrefix) {
    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 0u);
    }

    StableClient->WriteFile(TabletId, "a", "a content").GetValueSync();
    StableClient->WriteFile(TabletId, "b1", "b1 content").GetValueSync();
    StableClient->WriteFile(TabletId, "b2", "b2 content").GetValueSync();

    Client->RemovePrefix(TabletId, "b").GetValueSync();

    {
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u);
        EXPECT_EQ(files[0].Name, "1");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "1").GetValueSync().Value(), "b1 content");
        EXPECT_EQ(files[1].Name, "2");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "2").GetValueSync().Value(), "b2 content");
        EXPECT_EQ(files[2].Name, "a");
        EXPECT_EQ(StableClient->ReadFile(TabletId, "a").GetValueSync().Value(), "a content");
    }
}

TEST_F(TKvClientTabletTest, TabletOperationsBatch) {
    Client->BatchRequest(
        TKikimrKvBatchRequest(TabletId)
            .WriteFile("l1", "log 1")
            .WriteFile("l2", "log 2")
            .WriteFile("l3", "log 3")).GetValueSync();

    TVector<TFileInfo> files;
    files = StableClient->ListFiles(TabletId).GetValueSync().Value();
    ASSERT_EQ(files.size(), 3u);
    EXPECT_EQ(files[0].Name, "l1");
    EXPECT_EQ(files[1].Name, "l2");
    EXPECT_EQ(files[2].Name, "l3");

    Client->BatchRequest(
        TKikimrKvBatchRequest(TabletId)
            .WriteFile("s1", "snapshot 1")
            .WriteFile("s2", "snapshot 2")
            .RemoveFiles(TKikimrKvRange{"l1", "l3"})).GetValueSync();

    files = StableClient->ListFiles(TabletId).GetValueSync().Value();
    ASSERT_EQ(files.size(), 3u);
    EXPECT_EQ(files[0].Name, "l3");
    EXPECT_EQ(files[1].Name, "s1");
    EXPECT_EQ(files[2].Name, "s2");

    auto result = Client->BatchRequest(
        TKikimrKvBatchRequest(TabletId)
            .ListFiles()
            .ReadFile("l3")
            .ReadFile("s1")
            .ReadFile("s2")).GetValueSync().Value();

    ASSERT_EQ(result.ListFileQueryResults.size(), 1u);
    ASSERT_EQ(result.ListFileQueryResults[0].size(), 3u);
    EXPECT_EQ(result.ListFileQueryResults[0][0].Name, "l3");
    EXPECT_EQ(result.ListFileQueryResults[0][1].Name, "s1");
    EXPECT_EQ(result.ListFileQueryResults[0][2].Name, "s2");
    ASSERT_EQ(result.ReadFileQueryResults.size(), 3u);
    EXPECT_EQ(result.ReadFileQueryResults[0], "log 3");
    EXPECT_EQ(result.ReadFileQueryResults[1], "snapshot 1");
    EXPECT_EQ(result.ReadFileQueryResults[2], "snapshot 2");
}

TEST_F(TKvClientTabletTest, TabletOperationsDeadline) {
    auto files = Client->ListFiles(TabletId, TDuration::Seconds(10).ToDeadLine()).GetValueSync().Value();
    EXPECT_TRUE(files.empty());

    auto status = Client->ListFiles(TabletId, TInstant::Now() - TDuration::Seconds(10)).ExtractValueSync();
    EXPECT_FALSE(status.Success());
    EXPECT_THAT(status.Error().Message(), ::testing::AnyOf(
            ::testing::HasSubstr("Deadline reached before processing the request"),
            ::testing::HasSubstr("Tablet request timed out")));
}

TEST_F(TKvClientTabletTest, BulkListFilesOverrunAllowed) {
    TFileNamesGenerator gen("BulkListFilesOverrunAllowed");
    constexpr i32 BatchCount = 100;
    constexpr i32 FilesInBatch = 100;
    constexpr i32 FilesCount = BatchCount * FilesInBatch;

    std::unique_ptr<TKikimrKvClientOptions> kvClientOptions{
        new TKikimrKvClientOptions{.LimitBytes = 64 * 1024, .AllowOverrun = true}
    };
    TKikimrKvClient kvClient(StableRpc.get(), std::move(kvClientOptions));

    for (i32 i = 0; i < BatchCount; i++) {
        TKikimrKvBatchRequest batch(TabletId);
        for (i32 j = 0; j < FilesInBatch; j++) {
            TString filename = gen.NextFileName();
            batch.WriteFile(filename, filename);
        }
        kvClient.BatchRequest(batch).GetValueSync();
        gen.NextSeries();
    }

    auto future = kvClient.ListFiles(TabletId);
    const auto& result = future.GetValueSync();
    ASSERT_TRUE(result.Success()) << "List files failed with error: " << result.Error().Message();
    ASSERT_EQ((size_t)FilesCount, result.Value().size());
    std::vector<bool> numbers(FilesCount, false);
    for (const auto& fileInfo: result.Value()) {
        auto [series, number] = gen.GetSeriesAndFileNumber(fileInfo.Name);
        if(0 <= series && series < BatchCount && 0 <= number && number < FilesInBatch) {
            const auto n = series * FilesInBatch + number;
            EXPECT_FALSE(numbers[n]);
            numbers[n] = true;
        }
    }

    EXPECT_TRUE(std::all_of(numbers.begin(), numbers.end(), [](bool n) { return n; }));
}

TEST_F(TKvClientTabletTest, BulkListFilesOverrunProhibited) {
    TFileNamesGenerator gen("BulkListFilesOverrunProhibited");
    constexpr i32 BatchCount = 100;
    constexpr i32 FilesInBatch = 100;
    constexpr i32 FilesCount = BatchCount * FilesInBatch;

    std::unique_ptr<TKikimrKvClientOptions> kvClientOptions{
        new TKikimrKvClientOptions{.LimitBytes = 64 * 1024, .AllowOverrun = false}
    };
    TKikimrKvClient kvClient(StableRpc.get(), std::move(kvClientOptions));

    for (i32 i = 0; i < BatchCount; i++) {
        TKikimrKvBatchRequest batch(TabletId);
        for (i32 j = 0; j < FilesInBatch; j++) {
            TString filename = gen.NextFileName();
            batch.WriteFile(filename, filename);
        }
        kvClient.BatchRequest(batch).GetValueSync();
        gen.NextSeries();
    }

    auto future = kvClient.ListFiles(TabletId);
    const auto& result = future.GetValueSync();
    ASSERT_TRUE(result.Success()) << "List files failed with error: " << result.Error().Message();
    ASSERT_LT(result.Value().size(), (size_t)FilesCount);
}
