#include "labels.h"

#include <util/generic/hash.h>

namespace NSolomon::NLabels {

TLabels::TLabels(const TLabels& rhs)
    : Labels_{rhs.Labels_}
    , Size_{rhs.Size_}
{
    if (rhs.OwnsStorage_) {
        EnsureStorageHeap();
    }
    if (rhs.OwnsStrings_) {
        EnsureStringsHeap();
    }
}

TLabels::TLabels(TLabels&& rhs) noexcept {
    std::swap(Labels_, rhs.Labels_);
    std::swap(Size_, rhs.Size_);
    std::swap(OwnsStorage_, rhs.OwnsStorage_);
    std::swap(OwnsStrings_, rhs.OwnsStrings_);
}

TLabels& TLabels::operator=(const TLabels& rhs) {
    TLabels copy{rhs};
    std::swap(*this, copy);
    return *this;
}

TLabels& TLabels::operator=(TLabels&& rhs) noexcept {
    std::swap(Labels_, rhs.Labels_);
    std::swap(Size_, rhs.Size_);
    std::swap(OwnsStorage_, rhs.OwnsStorage_);
    std::swap(OwnsStrings_, rhs.OwnsStrings_);
    return *this;
}

TLabels::~TLabels() {
    if (OwnsStrings_) {
        for (ui16 i = 0; i < Size_; ++i) {
            delete[] Labels_[i].first;
            delete[] Labels_[i].second;
        }
    }

    if (OwnsStorage_) {
        delete[] Labels_;
    }
}

ui64 TLabels::Hash() const {
    return TLabelsView{*this}.Hash();
}

void TLabels::EnsureHeap() noexcept {
    EnsureStringsHeap();
}

void TLabels::FilterKeys(const THashSet<TString>& filter) {
    EnsureStorageHeap();

    auto head = const_cast<TPair*>(Labels_); // we own Labels_ so const cast is safe

    for (auto it = Labels_, end = Labels_ + Size_; it != end; ++it) {
        if (filter.contains(it->first)) {
            *head++ = *it;
        } else {
            Size_--;
        }
    }
}

void TLabels::EnsureStorageHeap() noexcept {
    if (!OwnsStorage_) {
        auto labels = new TPair[Size_];
        Copy(Labels_, Labels_ + Size_, labels);
        Labels_ = labels;
        OwnsStorage_ = true;
    }
}

void TLabels::EnsureStringsHeap() noexcept {
    EnsureStorageHeap();

    if (!OwnsStrings_) {
        auto labels = const_cast<TPair*>(Labels_); // we own Labels_ so const cast is safe
        for (ui16 i = 0; i < Size_; ++i) {
            labels[i].first = strcpy(new char[strlen(labels[i].first) + 1], labels[i].first);
            labels[i].second = strcpy(new char[strlen(labels[i].second) + 1], labels[i].second);
        }
        OwnsStrings_ = true;
    }
}

void TLabels::EnsureSorted() noexcept {
    EnsureStorageHeap();

    auto labels = const_cast<TPair*>(Labels_); // we own Labels_ so const cast is safe

    std::sort(labels, labels + Size_, [](auto lhs, auto rhs) {
        return strcmp(lhs.first, rhs.first) < 0;
    });

    auto end = std::unique(labels, labels + Size_, [](auto lhs, auto rhs) {
        return strcmp(lhs.first, rhs.first) == 0;
    });

    if (OwnsStrings_) {
        for (auto it = end; it != labels + Size_; ++it) {
            delete[] it->first;
            delete[] it->second;
        }
    }

    Size_ = std::distance(labels, end);
}

bool operator==(const TLabels& lhs, const TLabels& rhs) {
    return TLabelsView{lhs} == TLabelsView{rhs};
}

bool operator!=(const TLabels& lhs, const TLabels& rhs) {
    return TLabelsView{lhs} != TLabelsView{rhs};
}

std::ostream& operator<<(std::ostream& os, const TLabels& labels) {
    return os << TLabelsView{labels};
}

IOutputStream& operator<<(IOutputStream& os, const TLabels& labels) {
    return os << TLabelsView{labels};
}

TLabels::TLabels(TPtr labels, ui16 size, bool ownsStorage, bool ownsStrings)
    : Labels_{labels}
    , Size_{size}
    , OwnsStorage_{ownsStorage}
    , OwnsStrings_{ownsStrings}
{
}

ui64 TLabelsView::Hash() const {
    ui64 hash = ComputeHash(Size());
    for (auto [k, v]: *this) {
        hash = CombineHashes(hash, ComputeHash(TStringBuf(k)));
        hash = CombineHashes(hash, ComputeHash(TStringBuf(v)));
    }
    return hash;
}

bool operator==(const TLabelsView& lhs, const TLabelsView& rhs) {
    return std::equal(
        lhs.begin(), lhs.end(),
        rhs.begin(), rhs.end(),
        [](const TLabelsView::TPair& lhs, const TLabelsView::TPair& rhs) {
            return strcmp(lhs.first, rhs.first) == 0 && strcmp(lhs.second, rhs.second) == 0;
        });
}

bool operator!=(const TLabelsView& lhs, const TLabelsView& rhs) {
    return !(lhs == rhs);
}

std::ostream& operator<<(std::ostream& os, const TLabelsView& labels) {
    os << '{';
    TStringBuf sep;
    for (auto [k, v]: labels) {
        os << sep << k << ": " << v;
        sep = ", ";
    }
    os << '}';
    return os;
}

IOutputStream& operator<<(IOutputStream& os, const TLabelsView& labels) {
    os << '{';
    TStringBuf sep;
    for (auto [k, v]: labels) {
        os << sep << k << ": " << v;
        sep = ", ";
    }
    os << '}';
    return os;
}

} // namespace NSolomon::NLabels
