#pragma once

#include <util/generic/algorithm.h>
#include <util/generic/hash_set.h>
#include <util/generic/ylimits.h>
#include <util/system/types.h>
#include <util/system/yassert.h>

#include <ostream>

namespace NSolomon::NLabels {

class TLabelsView;

/**
 * Represents metric labels, i.e. a sequence of key-value pairs of strings.
 *
 * This class stores labels sorted lexicographically by keys.
 *
 * Each key and value is represented by a pointer to a c-style null-terminated string.
 *
 * This class also supports mixed allocation strategy. That is, if your labels are allocated in an arena,
 * you don't have to copy it into the heap. Instead, you can construct `TLabels` with pooled data,
 * and it will remember not to call any destructors.
 */
class TLabels {
public:
    using TPair = std::pair<const char*, const char*>;
    using TPtr = const TPair*;

public:
    /**
     * Create an instance of `TLabels` that doesn't own its contents.
     *
     * NOTE: Given range should contain unique keys sorted lexicographically.
     *
     * This is essentially an array view. Contents of the returned object will be valid
     * for as long as the given array stays valid.
     */
    static inline TLabels UnownedSorted(TPtr begin, TPtr end);
    static inline TLabels UnownedSorted(TPtr begin, ui16 size);

    /**
     * Create an instance of `TLabels` that owns its array of string pairs, but does not own the strings themselves.
     *
     * NOTE: If using `OwnedStorageSorted`, given range should contain unique keys sorted lexicographically.
     *
     * This is essentially a shallow copy of the given range. Contents of the returned object will be valid for as long
     * as the keys and values from the given range stay valid.
     */
    template <typename TIt>
    static inline TLabels OwnedStorage(TIt begin, TIt end);
    template <typename TIt>
    static inline TLabels OwnedStorageSorted(TIt begin, TIt end);
    static inline TLabels OwnedStorage(TPtr begin, ui16 size);
    static inline TLabels OwnedStorageSorted(TPtr begin, ui16 size);
    static inline TLabels OwnedStorage(std::initializer_list<TPair> labels);
    static inline TLabels OwnedStorageSorted(std::initializer_list<TPair> labels);

    /**
     * Create an instance of `TLabels` that owns both its array of string pairs and the strings themselves.
     *
     * NOTE: If using `OwnedStringsSorted`, given range should contain unique keys sorted lexicographically.
     *
     * This is essentially a deep copy of the given range. Contents of the returned object will be valid for as long
     * as it exists.
     */
    template <typename TIt>
    static inline TLabels OwnedStrings(TIt begin, TIt end);
    template <typename TIt>
    static inline TLabels OwnedStringsList(TIt begin, TIt end);
    template <typename TIt>
    static inline TLabels OwnedStringsSorted(TIt begin, TIt end);
    static inline TLabels OwnedStrings(TPtr begin, ui16 size);
    static inline TLabels OwnedStringsSorted(TPtr begin, ui16 size);
    static inline TLabels OwnedStrings(std::initializer_list<TPair> labels);
    static inline TLabels OwnedStringsSorted(std::initializer_list<TPair> labels);

private:
    TLabels(TPtr labels, ui16 size, bool ownsStorage, bool ownsStrings);

public:
    /**
     * Create empty labels.
     */
    TLabels() = default;

    /**
     * Copy labels. If `rhs` does not own its storage, new copy will now own the storage either.
     */
    TLabels(const TLabels& rhs);

    /**
     * Move labels.
     */
    TLabels(TLabels&& rhs) noexcept;

    /**
     * Copy-assign labels. If `rhs` does not own its storage, new copy will now own the storage either.
     */
    TLabels& operator=(const TLabels&);

    /**
     * Move-assign labels.
     */
    TLabels& operator=(TLabels&&) noexcept;

    /**
     * Free memory if `ownsStorage` or `ownsStrings` is set.
     */
    ~TLabels();

public:
    /**
     * Get number of labels.
     */
    ui16 Size() const {
        return Size_;
    }

    /**
     * Check if there is at least a single label in this class.
     */
    bool Empty() const {
        return Size_ == 0;
    }

    /**
     * Calculate hash value of all labels.
     */
    ui64 Hash() const;

    /**
     * Ensure that labels are stored in the heap, copy them to the heap if necessary.
     */
    void EnsureHeap() noexcept;

    /**
     * Remove all keys that are not found in this set. Moves storage to heap if necessary.
     */
    void FilterKeys(const THashSet<TString>& filter);

public:
    /**
     * Get label by index.
     */
    TPair operator[](size_t idx) {
        Y_ASSERT(idx < Size_);
        return Labels_[idx];
    }

public:
    TPtr begin() const {
        return Begin();
    }
    TPtr Begin() const {
        return Labels_;
    }
    TPtr end() const {
        return End();
    }
    TPtr End() const {
        return Labels_ + Size_;
    }

public:
    friend bool operator==(const TLabels& lhs, const TLabels& rhs);
    friend bool operator!=(const TLabels& lhs, const TLabels& rhs);
    friend std::ostream& operator<<(std::ostream& os, const TLabels& labels);
    friend IOutputStream& operator<<(IOutputStream& os, const TLabels& labels);

private:
    void EnsureStorageHeap() noexcept;
    void EnsureStringsHeap() noexcept;
    void EnsureSorted() noexcept;

private:
    TPtr Labels_ = nullptr;
    ui16 Size_ = 0;
    bool OwnsStorage_ = false; // if true, we need to call delete on Labels_
    bool OwnsStrings_ = false; // if true, we need to call delete on all Labels_ items
};

/**
 * A trivially destructible version of `TLabels` that doesn't own its contents.
 */
class TLabelsView {
public:
    using TPair = std::pair<const char*, const char*>;
    using TPtr = const TPair*;

public:
    TLabelsView(const TLabels& labels)
        : Labels_{labels.Begin()}
        , Size_{labels.Size()}
    {
    }

public:
    /**
     * Get number of labels.
     */
    ui16 Size() const {
        return Size_;
    }

    /**
     * Check if there is at least a single label in this class.
     */
    bool Empty() const {
        return Size_ == 0;
    }

    /**
     * Calculate hash value of all labels.
     */
    ui64 Hash() const;

public:
    /**
     * Get label by index.
     */
    TPair operator[](size_t idx) {
        Y_ASSERT(idx < Size_);
        return Labels_[idx];
    }

public:
    TPtr begin() const {
        return Begin();
    }
    TPtr Begin() const {
        return Labels_;
    }
    TPtr end() const {
        return End();
    }
    TPtr End() const {
        return Labels_ + Size_;
    }

public:
    friend bool operator==(const TLabelsView& lhs, const TLabelsView& rhs);
    friend bool operator!=(const TLabelsView& lhs, const TLabelsView& rhs);
    friend std::ostream& operator<<(std::ostream& os, const TLabelsView& labels);
    friend IOutputStream& operator<<(IOutputStream& os, const TLabelsView& labels);

private:
    TPtr Labels_ = nullptr;
    ui16 Size_ = 0;
};

} // namespace NSolomon::NLabels

template <>
struct THash<NSolomon::NLabels::TLabels> {
    size_t operator()(const NSolomon::NLabels::TLabels& labels) const noexcept {
        return labels.Hash();
    }
};

template <>
struct THash<NSolomon::NLabels::TLabelsView> {
    size_t operator()(const NSolomon::NLabels::TLabelsView& labels) const noexcept {
        return labels.Hash();
    }
};

namespace NSolomon::NLabels {

TLabels TLabels::UnownedSorted(TLabels::TPtr begin, TLabels::TPtr end) {
    auto size = std::distance(begin, end);
    Y_ASSERT(size <= Max<ui16>());
    return UnownedSorted(begin, size);
}

TLabels TLabels::UnownedSorted(TLabels::TPtr begin, ui16 size) {
    return TLabels{begin, size, false, false};
}

template <typename TIt>
TLabels TLabels::OwnedStorage(TIt begin, TIt end) {
    auto res = OwnedStorageSorted(begin, end);
    res.EnsureSorted();
    return res;
}

template <typename TIt>
TLabels TLabels::OwnedStorageSorted(TIt begin, TIt end) {
    auto size = std::distance(begin, end);
    Y_ASSERT(size <= Max<ui16>());

    auto data = new TPair[size];
    Copy(begin, end, data);

    return TLabels{data, static_cast<ui16>(size), true, false};
}

TLabels TLabels::OwnedStorage(TLabels::TPtr begin, ui16 size) {
    return OwnedStorage(begin, begin + size);
}

TLabels TLabels::OwnedStorageSorted(TLabels::TPtr begin, ui16 size) {
    return OwnedStorageSorted(begin, begin + size);
}

TLabels TLabels::OwnedStorage(std::initializer_list<TPair> labels) {
    return OwnedStorage(labels.begin(), labels.end());
}

TLabels TLabels::OwnedStorageSorted(std::initializer_list<TPair> labels) {
    return OwnedStorageSorted(labels.begin(), labels.end());
}

template <typename TIt>
TLabels TLabels::OwnedStrings(TIt begin, TIt end) {
    auto res = OwnedStringsSorted(begin, end);
    res.EnsureSorted();
    return res;
}

template <typename TIt>
TLabels TLabels::OwnedStringsList(TIt begin, TIt end) {
    auto size = std::distance(begin, end);
    Y_ASSERT(size <= Max<ui16>());
    Y_ASSERT(size % 2 == 0);
    size /= 2;

    auto* data = new TPair[size];
    size_t i = 0;
    for (auto it = begin; it != end; ) {
        data[i].first = (it++)->data();
        data[i].second = (it++)->data();
        i++;
    }

    TLabels labels{data, static_cast<ui16>(size), true, false};
    labels.EnsureStringsHeap();
    labels.EnsureSorted();
    return labels;
}

template <typename TIt>
TLabels TLabels::OwnedStringsSorted(TIt begin, TIt end) {
    auto res = OwnedStorageSorted(begin, end);
    res.EnsureStringsHeap();
    return res;
}

TLabels TLabels::OwnedStrings(TLabels::TPtr begin, ui16 size) {
    return OwnedStrings(begin, begin + size);
}

TLabels TLabels::OwnedStringsSorted(TLabels::TPtr begin, ui16 size) {
    return OwnedStringsSorted(begin, begin + size);
}

TLabels TLabels::OwnedStrings(std::initializer_list<TPair> labels) {
    return OwnedStrings(labels.begin(), labels.end());
}

TLabels TLabels::OwnedStringsSorted(std::initializer_list<TPair> labels) {
    return OwnedStringsSorted(labels.begin(), labels.end());
}

} // namespace NSolomon::NLabels
