#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/labels/labels.h>

#include <util/generic/vector.h>

using namespace NSolomon::NLabels;

void ExpectLabelsEq(const TLabels& labels, std::initializer_list<TLabels::TPair> expected) {
    auto labelsBegin = labels.begin();
    auto labelsEnd = labels.end();

    auto expectedBegin = expected.begin();
    auto expectedEnd = expected.end();

    size_t i = 0;

    while (labelsBegin != labelsEnd && expectedBegin != expectedEnd) {
        EXPECT_STREQ(labelsBegin->first, expectedBegin->first) << "key differs at label #" << i;
        EXPECT_STREQ(labelsBegin->second, expectedBegin->second) << "value differs at label #" << i;
        labelsBegin++;
        expectedBegin++;
    }

    if (labelsBegin != labelsEnd) {
        FAIL()
            << "got more labels than expected; first extra label is "
            << "{" << labelsBegin->first << ": " << labelsBegin->second << "}";
    }

    if (expectedBegin != expectedEnd) {
        FAIL()
            << "got less labels than expected; first missing label is "
            << "{" << expectedBegin->first << ": " << expectedBegin->second << "}";
    }
}

TEST(Labels, Size) {
    {
        auto labels = TLabels::OwnedStorageSorted({});
        EXPECT_TRUE(labels.Empty());
        EXPECT_EQ(labels.Size(), 0u);
    }

    {
        auto labels = TLabels::OwnedStorageSorted({{"a", "1"}});
        EXPECT_FALSE(labels.Empty());
        EXPECT_EQ(labels.Size(), 1u);
    }

    {
        auto labels = TLabels::OwnedStorageSorted({{"a", "1"}, {"b", "2"}});
        EXPECT_FALSE(labels.Empty());
        EXPECT_EQ(labels.Size(), 2u);
    }
}

TEST(Labels, Iter) {
    {
        auto labels = TLabels::OwnedStorageSorted({});
        ExpectLabelsEq(labels, {});
    }

    {
        auto labels = TLabels::OwnedStorageSorted({{"a", "1"}});
        ExpectLabelsEq(labels, {{"a", "1"}});
    }

    {
        auto labels = TLabels::OwnedStorageSorted({{"a", "1"}, {"b", "2"}});
        ExpectLabelsEq(labels, {{"a", "1"}, {"b", "2"}});
    }
}

TEST(Labels, HashAndEq) {
    {
        auto l = TLabels::OwnedStorageSorted({});
        auto r = TLabels::OwnedStorageSorted({});
        EXPECT_EQ(l, r);
        EXPECT_EQ(l.Hash(), r.Hash());
    }

    {
        auto l = TLabels::OwnedStorageSorted({{"a", "1"}});
        auto r = TLabels::OwnedStorageSorted({{"a", "1"}});
        EXPECT_EQ(l, r);
        EXPECT_EQ(l.Hash(), r.Hash());
    }

    {
        auto l = TLabels::OwnedStorageSorted({{"a", "1"}, {"b", "2"}});
        auto r = TLabels::OwnedStorageSorted({{"a", "1"}, {"b", "2"}});
        EXPECT_EQ(l, r);
        EXPECT_EQ(l.Hash(), r.Hash());
    }

    {
        auto l = TLabels::OwnedStorageSorted({});
        auto r = TLabels::OwnedStorageSorted({{"a", "1"}});
        EXPECT_NE(l, r);
        EXPECT_NE(l.Hash(), r.Hash());
    }

    {
        auto l = TLabels::OwnedStorageSorted({{"a", "1"}});
        auto r = TLabels::OwnedStorageSorted({{"a", "2"}});
        EXPECT_NE(l, r);
        EXPECT_NE(l.Hash(), r.Hash());
    }

    {
        auto l = TLabels::OwnedStorageSorted({{"a", "1"}});
        auto r = TLabels::OwnedStorageSorted({{"b", "1"}});
        EXPECT_NE(l, r);
        EXPECT_NE(l.Hash(), r.Hash());
    }
}

TEST(Labels, Unowned) {
    TVector<TLabels::TPair> contents{{"a", "1"}, {"b", "2"}};

    auto labels = TLabels::UnownedSorted(contents.begin(), contents.end());

    ExpectLabelsEq(labels, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labels.begin(), contents.begin());
    EXPECT_EQ(labels.end(), contents.end());

    TLabels labelsCopyConstructed{labels};

    ExpectLabelsEq(labelsCopyConstructed, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsCopyConstructed.begin(), contents.begin());
    EXPECT_EQ(labelsCopyConstructed.end(), contents.end());

    TLabels labelsMoveConstructed{std::move(labelsCopyConstructed)};

    ExpectLabelsEq(labelsMoveConstructed, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsMoveConstructed.begin(), contents.begin());
    EXPECT_EQ(labelsMoveConstructed.end(), contents.end());

    TLabels labelsCopyAssigned;
    labelsCopyAssigned = labels;

    ExpectLabelsEq(labelsCopyAssigned, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsCopyAssigned.begin(), contents.begin());
    EXPECT_EQ(labelsCopyAssigned.end(), contents.end());

    TLabels labelsMoveAssigned;
    labelsMoveAssigned = std::move(labelsCopyAssigned);

    ExpectLabelsEq(labelsMoveAssigned, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsMoveAssigned.begin(), contents.begin());
    EXPECT_EQ(labelsMoveAssigned.end(), contents.end());
}

TEST(Labels, StorageHeap) {
    auto ak = "a";
    auto av = "1";
    auto bk = "b";
    auto bv = "2";
    TVector<TLabels::TPair> contents{{ak, av}, {bk, bv}};

    auto labels = TLabels::OwnedStorageSorted(contents.begin(), contents.end());

    contents.clear();
    contents.shrink_to_fit();

    ExpectLabelsEq(labels, {{"a", "1"}, {"b", "2"}});

    EXPECT_NE(labels.begin(), contents.begin());
    EXPECT_NE(labels.end(), contents.end());
    EXPECT_EQ(labels[0].first, ak);
    EXPECT_EQ(labels[0].second, av);
    EXPECT_EQ(labels[1].first, bk);
    EXPECT_EQ(labels[1].second, bv);

    TLabels labelsCopyConstructed{labels};

    ExpectLabelsEq(labelsCopyConstructed, {{"a", "1"}, {"b", "2"}});

    EXPECT_NE(labelsCopyConstructed.begin(), labels.begin());
    EXPECT_NE(labelsCopyConstructed.end(), labels.end());
    EXPECT_EQ(labelsCopyConstructed[0].first, ak);
    EXPECT_EQ(labelsCopyConstructed[0].second, av);
    EXPECT_EQ(labelsCopyConstructed[1].first, bk);
    EXPECT_EQ(labelsCopyConstructed[1].second, bv);

    auto labelsCopyConstructedBegin = labelsCopyConstructed.begin();
    auto labelsCopyConstructedEnd = labelsCopyConstructed.end();

    TLabels labelsMoveConstructed{std::move(labelsCopyConstructed)};

    ExpectLabelsEq(labelsMoveConstructed, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsCopyConstructed.begin(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsCopyConstructed.end(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsMoveConstructed.begin(), labelsCopyConstructedBegin);
    EXPECT_EQ(labelsMoveConstructed.end(), labelsCopyConstructedEnd);
    EXPECT_EQ(labelsMoveConstructed[0].first, ak);
    EXPECT_EQ(labelsMoveConstructed[0].second, av);
    EXPECT_EQ(labelsMoveConstructed[1].first, bk);
    EXPECT_EQ(labelsMoveConstructed[1].second, bv);

    TLabels labelsCopyAssigned;
    labelsCopyAssigned = labels;

    ExpectLabelsEq(labelsCopyAssigned, {{"a", "1"}, {"b", "2"}});

    EXPECT_NE(labelsCopyAssigned.begin(), labels.begin());
    EXPECT_NE(labelsCopyAssigned.end(), labels.end());
    EXPECT_EQ(labelsCopyAssigned[0].first, ak);
    EXPECT_EQ(labelsCopyAssigned[0].second, av);
    EXPECT_EQ(labelsCopyAssigned[1].first, bk);
    EXPECT_EQ(labelsCopyAssigned[1].second, bv);

    auto labelsCopyAssignedBegin = labelsCopyAssigned.begin();
    auto labelsCopyAssignedEnd = labelsCopyAssigned.end();

    TLabels labelsMoveAssigned;
    labelsMoveAssigned = std::move(labelsCopyAssigned);

    ExpectLabelsEq(labelsMoveAssigned, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsCopyAssigned.begin(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsCopyAssigned.end(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsMoveAssigned.begin(), labelsCopyAssignedBegin);
    EXPECT_EQ(labelsMoveAssigned.end(), labelsCopyAssignedEnd);
    EXPECT_EQ(labelsMoveAssigned[0].first, ak);
    EXPECT_EQ(labelsMoveAssigned[0].second, av);
    EXPECT_EQ(labelsMoveAssigned[1].first, bk);
    EXPECT_EQ(labelsMoveAssigned[1].second, bv);
}

TEST(Labels, StorageHeapSort) {
    auto labels = TLabels::OwnedStorage({{"b", "1"}, {"a", "2"}, {"b", "3"}});

    ExpectLabelsEq(labels, {{"a", "2"}, {"b", "1"}});
}

TEST(Labels, StringsHeap) {
    auto ak = "a";
    auto av = "1";
    auto bk = "b";
    auto bv = "2";
    TVector<TLabels::TPair> contents{{ak, av}, {bk, bv}};

    auto labels = TLabels::OwnedStringsSorted(contents.begin(), contents.end());

    contents.clear();
    contents.shrink_to_fit();

    ExpectLabelsEq(labels, {{"a", "1"}, {"b", "2"}});

    EXPECT_NE(labels.begin(), contents.begin());
    EXPECT_NE(labels.end(), contents.end());
    EXPECT_NE(labels[0].first, ak);
    EXPECT_NE(labels[0].second, av);
    EXPECT_NE(labels[1].first, bk);
    EXPECT_NE(labels[1].second, bv);

    TLabels labelsCopyConstructed{labels};

    ExpectLabelsEq(labelsCopyConstructed, {{"a", "1"}, {"b", "2"}});

    EXPECT_NE(labelsCopyConstructed.begin(), labels.begin());
    EXPECT_NE(labelsCopyConstructed.end(), labels.end());
    EXPECT_NE(labelsCopyConstructed[0].first, labels[0].first);
    EXPECT_NE(labelsCopyConstructed[0].second, labels[0].second);
    EXPECT_NE(labelsCopyConstructed[1].first, labels[1].first);
    EXPECT_NE(labelsCopyConstructed[1].second, labels[1].second);

    auto labelsCopyConstructedBegin = labelsCopyConstructed.begin();
    auto labelsCopyConstructedEnd = labelsCopyConstructed.end();

    TLabels labelsMoveConstructed{std::move(labelsCopyConstructed)};

    ExpectLabelsEq(labelsMoveConstructed, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsCopyConstructed.begin(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsCopyConstructed.end(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsMoveConstructed.begin(), labelsCopyConstructedBegin);
    EXPECT_EQ(labelsMoveConstructed.end(), labelsCopyConstructedEnd);

    TLabels labelsCopyAssigned;
    labelsCopyAssigned = labels;

    ExpectLabelsEq(labelsCopyAssigned, {{"a", "1"}, {"b", "2"}});

    EXPECT_NE(labelsCopyAssigned.begin(), labels.begin());
    EXPECT_NE(labelsCopyAssigned.end(), labels.end());
    EXPECT_NE(labelsCopyAssigned[0].first, labels[0].first);
    EXPECT_NE(labelsCopyAssigned[0].second, labels[0].second);
    EXPECT_NE(labelsCopyAssigned[1].first, labels[1].first);
    EXPECT_NE(labelsCopyAssigned[1].second, labels[1].second);

    auto labelsCopyAssignedBegin = labelsCopyAssigned.begin();
    auto labelsCopyAssignedEnd = labelsCopyAssigned.end();

    TLabels labelsMoveAssigned;
    labelsMoveAssigned = std::move(labelsCopyAssigned);

    ExpectLabelsEq(labelsMoveAssigned, {{"a", "1"}, {"b", "2"}});

    EXPECT_EQ(labelsCopyAssigned.begin(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsCopyAssigned.end(), nullptr);  // NOLINT(bugprone-use-after-move)
    EXPECT_EQ(labelsMoveAssigned.begin(), labelsCopyAssignedBegin);
    EXPECT_EQ(labelsMoveAssigned.end(), labelsCopyAssignedEnd);
}

TEST(Labels, StringsHeapSort) {
    auto labels = TLabels::OwnedStrings({{"b", "1"}, {"a", "2"}, {"b", "3"}});

    ExpectLabelsEq(labels, {{"a", "2"}, {"b", "1"}});
}

TEST(Labels, OwnedStringsList) {
    std::vector<TString> strings = {"a", "1", "b", "2", "c", "3"};

    auto labels = TLabels::OwnedStringsList(strings.begin(), strings.end());
    EXPECT_EQ(labels.Size(), 3u);

    EXPECT_STREQ(labels[0].first, "a");
    EXPECT_STREQ(labels[0].second, "1");

    EXPECT_STREQ(labels[1].first, "b");
    EXPECT_STREQ(labels[1].second, "2");

    EXPECT_STREQ(labels[2].first, "c");
    EXPECT_STREQ(labels[2].second, "3");
}
