#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/labels/validate.cpp>

#include <util/generic/vector.h>

using namespace NSolomon;

template <class TLabelsValidatorImpl>
void TestValidate(
    const TVector<TString>& validNames, const TVector<TString>& invalidNames,
    const TVector<TString>& validValues, const TVector<TString>& invalidValues)
{
    for (auto& labelName: invalidNames) {
        EXPECT_EQ(TLabelsValidatorImpl::ValidateLabelNameSafe(labelName), false);
        EXPECT_THROW(TLabelsValidatorImpl::ValidateLabelName(labelName), yexception);
    }

    for (auto& labelName: validNames) {
        EXPECT_EQ(TLabelsValidatorImpl::ValidateLabelNameSafe(labelName), true);
        EXPECT_NO_THROW(TLabelsValidatorImpl::ValidateLabelName(labelName));
    }

    for (auto& labelValue: invalidValues) {
        EXPECT_EQ(TLabelsValidatorImpl::ValidateLabelValueSafe(labelValue), false);
        EXPECT_THROW(TLabelsValidatorImpl::ValidateLabelValue(labelValue), yexception);
    }

    for (auto& labelValue: validValues) {
        EXPECT_EQ(TLabelsValidatorImpl::ValidateLabelValueSafe(labelValue), true);
        EXPECT_NO_THROW(TLabelsValidatorImpl::ValidateLabelValue(labelValue));
    }
}

TEST(Validate, NotStrict) {
    TVector<TString> validNames{
        "0",
        "-",
        "calc-udf",
        "cswkdfugjbsjv",
        "-0123456789qwertyuiopdfgkzvmQWERTYASDFGHJZXCVBNM_",
    };

    TVector<TString> invalidNames{
        "!asdffr",
        "",
        "+",
    };

    TVector<TString> validValues = {
        "@123.-",
        "anythingGoes:123_ 345!!?*&"
    };
    TVector<TString> invalidValues = {
        "",
    };

    TestValidate<TLabelsValidator>(validNames, invalidNames, validValues, invalidValues);
}

TEST(Validate, Strict) {
    TVector<TString> validNames{
        "cswkdfugjbsjv",
    };

    TVector<TString> invalidNames{
        "0",
        "-",
        "!asdffr",
        "",
        "+",
        "-0123456789qwertyuiopdfgkzvmQWERTYASDFGHJZXCVBNM_",
        "calc-udf",
    };

    TVector<TString> validValues = {
        "@123.-",
    };
    TVector<TString> invalidValues = {
        "anythingGoes:123_ 345!!?*&"
    };

    TestValidate<TLabelsValidatorStrict>(validNames, invalidNames, validValues, invalidValues);
}
