#include "validate.h"

#include <util/generic/yexception.h>

namespace NSolomon {
namespace {

THolder<re2::RE2> CompileRE2WithCheck(const std::string& pattern) {
    THolder<re2::RE2> re(new re2::RE2(pattern));

    Y_ENSURE(re->ok(), "Unable to compile regex " << pattern << ": " << re->error());
    return re;
}

// Constructed by hand from source code: https://nda.ya.ru/3Vk3yy
const std::string LABEL_NAME_PATTERN = R"(^[a-zA-Z0-9-._/]{1,50}$)";
THolder<re2::RE2> LABEL_NAME_RE = CompileRE2WithCheck(LABEL_NAME_PATTERN);

// Strict
const std::string STRICT_LABEL_NAME_PATTERN = R"(^[a-zA-Z][a-zA-Z0-9_.]{0,31}$)";
THolder<re2::RE2> STRICT_LABEL_NAME_RE = CompileRE2WithCheck(STRICT_LABEL_NAME_PATTERN);

const std::string STRICT_LABEL_VALUE_PATTERN = R"(^[ -!#-&(-)+->@-_a-{}-~]{1,200}$)";
THolder<re2::RE2> STRICT_LABEL_VALUE_RE = CompileRE2WithCheck(STRICT_LABEL_VALUE_PATTERN);

} // namespace

//////////////////////////////////////////////////////////
// TLabelsValidator
//////////////////////////////////////////////////////////

bool TLabelsValidator::ValidateLabelNameSafe(TStringBuf name) {
    if (!RE2::FullMatch({name.data(), name.size()}, *LABEL_NAME_RE)) {
        return false;
    }

    return true;
}

void TLabelsValidator::ValidateLabelName(TStringBuf name) {
    if (!ValidateLabelNameSafe(name)) {
        throw yexception() << "label name \"" << name << "\" doesn't match a pattern \"" << LABEL_NAME_PATTERN << "\"";
    }
}

bool TLabelsValidator::ValidateLabelValueSafe(TStringBuf value) {
    // Right now, only checks a value for emptiness
    return !value.empty();
}

void TLabelsValidator::ValidateLabelValue(TStringBuf value) {
    if (!ValidateLabelValueSafe(value)) {
        throw yexception() << "empty label value";
    }
}


//////////////////////////////////////////////////////////
// TLabelsValidatorStrict
//////////////////////////////////////////////////////////

bool TLabelsValidatorStrict::ValidateLabelNameSafe(TStringBuf name) {
    if (!RE2::FullMatch({name.data(), name.size()}, *STRICT_LABEL_NAME_RE)) {
        return false;
    }

    return true;
}

void TLabelsValidatorStrict::ValidateLabelName(TStringBuf name) {
    if (!ValidateLabelNameSafe(name)) {
        throw yexception() << "label name \"" << name << "\" doesn't match a pattern \"" << STRICT_LABEL_NAME_PATTERN << "\"";
    }
}

bool TLabelsValidatorStrict::ValidateLabelValueSafe(TStringBuf value) {
    if (!RE2::FullMatch({value.data(), value.size()}, *STRICT_LABEL_VALUE_RE)) {
        return false;
    }

    return true;
}

void TLabelsValidatorStrict::ValidateLabelValue(TStringBuf value) {
    if (!ValidateLabelValueSafe(value)) {
        throw yexception() << "label value \"" << value << "\" doesn't match a pattern \"" << STRICT_LABEL_VALUE_PATTERN << "\"";
    }
}

} // namespace NSolomon
