#include "limiter.h"

#include <atomic>

namespace NSolomon {

namespace {

class TLimiter: public ILimiter {
public:
    explicit TLimiter(size_t limit)
        : Limit_{limit}
        , Usage_{0}
    {
    }

public:
    size_t GetLimit() const override {
        return Limit_;
    }

    size_t GetUsage() const override {
        return Usage_.load(std::memory_order_relaxed);
    }

    bool TryAdd(size_t value) override {
        if (!Limit_) {
            Usage_.fetch_add(value, std::memory_order_relaxed);
        } else {
            size_t usage = GetUsage();
            do {
                if (usage + value > Limit_) {
                    return false;
                }
            } while (!Usage_.compare_exchange_weak(usage, usage + value, std::memory_order_relaxed));
        }

        return true;
    }

    bool TryInc() override {
        return TryAdd(1);
    }

    void Sub(size_t value) override {
        Usage_.fetch_sub(value, std::memory_order_relaxed);
    }

    void Dec() override {
        Sub(1);
    }

private:
    size_t Limit_;
    std::atomic_size_t Usage_;
};

class TFakeLimiter: public ILimiter {
public:
    size_t GetLimit() const override {
        return 0;
    }

    size_t GetUsage() const override {
        return 0;
    }

    bool TryAdd(size_t value) override {
        Y_UNUSED(value);
        return true;
    }

    bool TryInc() override {
        return TryAdd(1);
    }

    void Sub(size_t value) override {
        Y_UNUSED(value);
    }

    void Dec() override {
        Sub(1);
    }
};

} // namespace

ILimiterPtr CreateLimiter(size_t limit) {
    return MakeIntrusive<TLimiter>(limit);
}

ILimiterPtr CreateFakeLimiter() {
    return MakeIntrusive<TFakeLimiter>();
}

} // namespace NSolomon::NMemStore::NPersister
