#pragma once

#include <util/generic/ptr.h>

namespace NSolomon {

/**
 * Limits some resource usage.
 */
class ILimiter: public TThrRefBase {
public:
    /**
     * Get total resource limit.
     */
    virtual size_t GetLimit() const = 0;

    /**
     * Get the current resource usage.
     */
    virtual size_t GetUsage() const = 0;

    /**
     * Increase current usage if new value is less then the limit.
     *
     * Return `true` if current usage was increased, `false` if usage is too high.
     */
    [[nodiscard]]
    virtual bool TryAdd(size_t value) = 0;

    /**
     * Increase current usage by one if new value is less then the limit.
     *
     * Return `true` if current usage was increased, `false` if usage is too high.
     */
    [[nodiscard]]
    virtual bool TryInc() = 0;

    /**
     * Decrease current usage.
     */
    virtual void Sub(size_t value) = 0;

    /**
     * Decrease current usage by one.
     */
    virtual void Dec() = 0;
};

using ILimiterPtr = TIntrusivePtr<ILimiter>;

/**
 * Create a default limiter implementation that uses atomic variables to count usage.
 *
 * @param limit     maximal allowed resource usage. `0` means unlimited (if you need to monitor usage without
 *                  limiting it).
 */
ILimiterPtr CreateLimiter(size_t limit);

/**
 * Create a fake limiter that doesn't count anything. `GetLimit` and `GetUsage` always return zero.
 */
ILimiterPtr CreateFakeLimiter();

} // namespace NSolomon
