#include <solomon/libs/cpp/limiter/limiter.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>

#include <latch>
#include <thread>
#include <vector>

TEST(TLimiterTest, Fake) {
    auto limiter = NSolomon::CreateFakeLimiter();

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);

    ASSERT_TRUE(limiter->TryAdd(100));

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);

    ASSERT_TRUE(limiter->TryInc());

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);

    limiter->Sub(100);

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);

    limiter->Dec();

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);
}

TEST(TLimiterTest, Unlimited) {
    auto limiter = NSolomon::CreateLimiter(0);

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);

    ASSERT_TRUE(limiter->TryAdd(100));

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 100u);

    ASSERT_TRUE(limiter->TryInc());

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 101u);

    limiter->Sub(50);

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 51u);

    limiter->Sub(50);

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 1u);

    limiter->Dec();

    ASSERT_EQ(limiter->GetLimit(), 0u);
    ASSERT_EQ(limiter->GetUsage(), 0u);
}

TEST(TLimiterTest, Limited) {
    auto limiter = NSolomon::CreateLimiter(100);

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 0u);

    ASSERT_TRUE(limiter->TryAdd(50));

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 50u);

    ASSERT_TRUE(limiter->TryAdd(50));

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 100u);

    ASSERT_TRUE(!limiter->TryAdd(50));

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 100u);

    limiter->Sub(50);

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 50u);

    ASSERT_TRUE(limiter->TryAdd(50));

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 100u);

    limiter->Sub(100);

    ASSERT_EQ(limiter->GetLimit(), 100u);
    ASSERT_EQ(limiter->GetUsage(), 0u);
}

TEST(TLimiterTest, Threaded) {
    constexpr auto THR_NUM = 8;
    auto limiter = NSolomon::CreateLimiter(100);

    std::vector<std::thread> threads;
    std::latch startBarrier{THR_NUM};

    for (auto i = 0; i < THR_NUM; ++i) {
        threads.emplace_back([&] {
            startBarrier.arrive_and_wait();

            for (auto j = 0; j < 100; ++j) {
                auto n = RandomNumber<ui8>() % 100;
                if (limiter->TryAdd(n)) {
                    limiter->Sub(n);
                }
            }
        });
    }

    for (auto&& t: threads) {
        t.join();
    }

    ASSERT_LT(limiter->GetUsage(), limiter->GetLimit());
}
