#include "local_shard_provider.h"

#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>
#include <solomon/libs/cpp/conf_db/model/shard_config.h>
#include <solomon/libs/cpp/conf_db/puller/puller.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/slices/operations.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

namespace NSolomon {
namespace {

using namespace NActors;
using namespace NSolomon::NDb::NModel;
using namespace NSolomon::NSlicer::NApi;
using namespace NSolomon;

using yandex::monitoring::slicer::GetSlicesByHostResponse;

class TLocalShardProvider: public TActorBootstrapped<TLocalShardProvider> {
public:
    TLocalShardProvider(TActorId sliceletListener, TActorId configsPuller)
        : SliceletListener_{sliceletListener}
        , ConfigsPuller_{configsPuller}
    {
    }

    void Bootstrap() {
        Send(SliceletListener_, new NSlicerClient::TSliceletListenerEvents::TEvSubscribe);
        Send(ConfigsPuller_, new TConfigsPullerEvents::TSubscribe);

        Become(&TThis::Main);
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TLocalShardProviderEvents::TSubscribe, OnSubscribe);
            hFunc(NSlicerClient::TSliceletListenerEvents::TSlicesUpdate, OnSlices);
            hFunc(TConfigsPullerEvents::TConfigsResponse, OnShardConfigs);
        }
    }

private:
    void OnSubscribe(const TLocalShardProviderEvents::TSubscribe::TPtr& evPtr) {
        auto [_, isNew] = Subscribers_.emplace(evPtr->Sender);

        if (!isNew || Matched_.empty()) {
            return;
        }

        TVector<NDb::NModel::TShardConfig> added(::Reserve(Matched_.size()));
        for (const auto& [_, config]: Matched_) {
            added.emplace_back(config);
        }

        Send(evPtr->Sender, new TLocalShardProviderEvents::TLocalShards{
            {}, // removed
            {}, // updated
            std::move(added),
        });
    }

    void OnSlices(const NSlicerClient::TSliceletListenerEvents::TSlicesUpdate::TPtr& evPtr) {
        auto& update = evPtr->Get()->Update;

        MON_INFO(LocalShardProvider, "got slices update");
        MON_TRACE(LocalShardProvider, update);

        Slices_.clear();

        for (int i = 0; i != update.assigned_starts_size(); ++i) {
            Slices_.emplace(update.assigned_starts(i), update.assigned_ends(i));
        }

        MatchShardConfigsWithSlices();
    }

    void OnShardConfigs(TConfigsPullerEvents::TConfigsResponse::TPtr& evPtr) {
        MON_INFO(LocalShardProvider, "got configs update");

        Configs_ = std::move(evPtr->Get()->Configs.Shards);

        MatchShardConfigsWithSlices();
    }

    void MatchShardConfigsWithSlices() {
        absl::flat_hash_map<TNumId, NDb::NModel::TShardConfig> newMatch;

        if (!Slices_.empty()) {
            for (const auto& config: Configs_) {
                if (SlicesContain(Slices_, config.NumId)) {
                    newMatch[config.NumId] = config;
                }
            }
        }

        TVector<TNumId> removed;
        TVector<NDb::NModel::TShardConfig> updated;
        TVector<NDb::NModel::TShardConfig> added;

        for (const auto& [numId, config]: Matched_) {
            if (auto it = newMatch.find(numId); it == newMatch.end()) {
                removed.emplace_back(numId);
            } else if (it->second != config) {
                updated.emplace_back(it->second);
            }
        }

        for (const auto& [numId, config]: newMatch) {
            if (!Matched_.contains(numId)) {
                added.emplace_back(config);
            }
        }

        Matched_ = std::move(newMatch);

        if (removed.empty() && updated.empty() && added.empty()) {
            return;
        }

        size_t i = 0;
        for (auto sub: Subscribers_) {
            if (i < Subscribers_.size() - 1) {
                Send(sub, new TLocalShardProviderEvents::TLocalShards{removed, updated, added});  // NOLINT(bugprone-use-after-move)
            } else {
                Send(sub, new TLocalShardProviderEvents::TLocalShards{std::move(removed), std::move(updated), std::move(added)});
            }
        }
    }

private:
    TActorId SliceletListener_;
    TActorId ConfigsPuller_;
    absl::flat_hash_set<TActorId, THash<TActorId>> Subscribers_;
    TSlices Slices_;
    TVector<NDb::NModel::TShardConfig> Configs_;
    absl::flat_hash_map<TNumId, NDb::NModel::TShardConfig> Matched_;
};

} // namespace

std::unique_ptr<NActors::IActor> CreateLocalShardProvider(
        NActors::TActorId sliceletListener,
        NActors::TActorId configsPuller)
{
    return std::make_unique<TLocalShardProvider>(sliceletListener, configsPuller);
}

}
