#pragma once
/// Yet another wrapper for actor lib's logging interface.
///
/// These macros can look up the current actor system context, so you don't have to cary it around.
/// They also add useful info to log message, such as file and line of the macros invocation.
///
/// First argument for each macro is the name of a component from `NSolomon::ELogComponent`.
/// Second argument is a message as formed with `IOutputStream`'s `operator<<`.
///
/// For example:
///
/// ```
/// auto reqId = 0;
/// MON_INFO(GrpcApi, "current request ID is " << reqId);
/// ```
///
/// Or if you want to pass an actor context explicitly:
///
/// ```
/// auto* actorSystem = NActors::TActorContext::ActorSystem();
/// SomeAsyncOp([actorSystem]() {
///     // ...
///     MON_DEBUG_C(actorSystem, Component, "logging outside of an actor context");
///     // ...
/// });
/// ```

#include <solomon/libs/cpp/actors/config/log_component.pb.h>

#include <library/cpp/actors/core/log.h>

#include <util/system/src_location.h>

/// Log a tracemessage.
#define MON_TRACE(C, ...)                                                                   \
    MON_TRACE_C(::NActors::TActorContext::AsActorContext(), C, __VA_ARGS__)
#define MON_TRACE_C(context, C, ...)                                                        \
    LOG_TRACE_S((context), ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Log a debug message.
#define MON_DEBUG(C, ...)                                                                   \
    MON_DEBUG_C(::NActors::TActorContext::AsActorContext(), C, __VA_ARGS__)
#define MON_DEBUG_C(context, C, ...)                                                        \
    LOG_DEBUG_S((context), ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Log an info message.
#define MON_INFO(C, ...)                                                                    \
    MON_INFO_C(::NActors::TActorContext::AsActorContext(), C, __VA_ARGS__)
#define MON_INFO_C(context, C, ...)                                                         \
    LOG_INFO_S((context), ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Log a warning message.
#define MON_WARN(C, ...)                                                                    \
    MON_WARN_C(::NActors::TActorContext::AsActorContext(), C, __VA_ARGS__)
#define MON_WARN_C(context, C, ...)                                                         \
    LOG_WARN_S((context), ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Log an error message.
#define MON_ERROR(C, ...)                                                                   \
    MON_ERROR_C(::NActors::TActorContext::AsActorContext(), C, __VA_ARGS__)
#define MON_ERROR_C(context, C, ...)                                                        \
    LOG_ERROR_S((context), ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Log a critical message.
#define MON_CRIT(C, ...)                                                                    \
    MON_CRIT_C(::NActors::TActorContext::AsActorContext(), C, __VA_ARGS__)
#define MON_CRIT_C(context, C, ...)                                                         \
    LOG_CRIT_S((context), ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Log message with specified priority level.
#define MON_LOG(C, PRIORITY, ...)                                                                    \
    MON_LOG_C(::NActors::TActorContext::AsActorContext(), C, PRIORITY, __VA_ARGS__)
#define MON_LOG_C(context, C, PRIORITY, ...)                                                         \
    LOG_LOG_S((context), PRIORITY, ::NSolomon::ELogComponent::C, '[' << __LOCATION__ << "] " << __VA_ARGS__)

/// Check condition, if it doesn't hold, log a critical message. In debug build, panic after logging a message.
#ifndef NDEBUG
#define MON_ASSERT_OR_LOG(A, C, ...)                                                        \
do {                                                                                        \
    try {                                                                                   \
        if (Y_UNLIKELY(!(A))) {                                                             \
            MON_CRIT(C, "assert " #A " failed: " << __VA_ARGS__);                           \
            Y_FAIL("assert " #A " failed: " << __VA_ARGS__);                                \
        }                                                                                   \
    } catch (...) {                                                                         \
        MON_CRIT(C, "exception in assert " #A ": " << ::CurrentExceptionMessage());         \
        Y_FAIL("exception in assert " #A ": " << ::CurrentExceptionMessage());              \
    }                                                                                       \
} while (false)
#else
#define MON_ASSERT_OR_LOG(A, C, ...)                                                        \
do {                                                                                        \
    try {                                                                                   \
        if (Y_UNLIKELY(!(A))) {                                                             \
            MON_CRIT(C, "assert " #A " failed: " << __VA_ARGS__);                           \
        }                                                                                   \
    } catch (...) {                                                                         \
        MON_CRIT(C, "exception in assert " #A ": " << ::CurrentExceptionMessage());         \
    }                                                                                       \
} while (false)
#endif
