#pragma once

#include <solomon/libs/cpp/ewma/ewma.h>
#include <solomon/libs/cpp/ewma/ewma2.h>
#include <solomon/libs/cpp/shard_metrics/repository.h>
#include <solomon/libs/cpp/shard_metrics/resource_context.h>

inline constexpr auto AvgWindow = TDuration::Seconds(30);

namespace NSolomon {

// TODO: merge all TNumId definitions
using TNumId = ui32;

class TShardMeteringContext: public TResourceUsageContextBlank {
public:
    using TBase = TResourceUsageContextBlank;
    using TBase::TBase;

    TShardMeteringContext(TString project, TString shard) noexcept
        : TResourceUsageContextBlank{std::move(project), std::move(shard)}
        , EwmaMemoryUsage_{AvgWindow, TInstant::Now()}
    {
    }

    void Init(NMonitoring::IMetricFactory& f) override {
        CpuUsageUs = f.Gauge(NMonitoring::MakeLabels({{"sensor", "cpu.userTimeUs"}}));
        MemoryUsage = f.Gauge(NMonitoring::MakeLabels({{"sensor", "memory.Bytes"}}));
        NetRxBytes = f.Gauge(NMonitoring::MakeLabels({{"sensor", "network.rxBytes"}}));
        NetTxBytes = f.Gauge(NMonitoring::MakeLabels({{"sensor", "network.txBytes"}}));

        EwmaCpuUsage_ = NMonitoring::TEwmaMeter(NMonitoring::OneMinuteEwma(CpuUsageUs));
        EwmaNetRx_ = NMonitoring::TEwmaMeter(NMonitoring::OneMinuteEwma(NetRxBytes));
        EwmaNetTx_ = NMonitoring::TEwmaMeter(NMonitoring::OneMinuteEwma(NetTxBytes));
    }

    void SetMemoryBytes(ui64 val) override {
        EwmaMemoryUsage_.Add(TInstant::Now(), static_cast<double>(val));
        MemoryUsage->Set(EwmaMemoryUsage_.Value());
    }

    void AddNetworkRxBytes(ui64 val) override {
        EwmaNetRx_.Mark(val);
    }

    void AddNetworkTxBytes(ui64 val) override {
        EwmaNetTx_.Mark(val);
    }

    void AddCpuTime(TDuration cpuTime) override {
        EwmaCpuUsage_.Mark(cpuTime.MicroSeconds());
    }

    void Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const override {
        MemoryUsage->Set(EwmaMemoryUsage_.Value());

        // TODO(ivanzhukov@): make TickIfNeeded() in monlib public
        EwmaCpuUsage_.Get();
        EwmaNetRx_.Get();
        EwmaNetTx_.Get();

        TBase::Append(time, consumer);
    }

public:
    NMonitoring::IGauge* CpuUsageUs;
    NMonitoring::IGauge* MemoryUsage;
    NMonitoring::IGauge* NetRxBytes;
    NMonitoring::IGauge* NetTxBytes;
    TNumId ShardNumId{0};

private:
    mutable TAtomicEwma EwmaMemoryUsage_;
    mutable NMonitoring::TEwmaMeter EwmaCpuUsage_;
    mutable NMonitoring::TEwmaMeter EwmaNetRx_;
    mutable NMonitoring::TEwmaMeter EwmaNetTx_;
};

struct TShardMeteringMetricsRepository: public TRepository<TShardMeteringContext> {
};

} // namespace NSolomon
