#include "minidump.h"
#include "minidump_path.h"
#include "resolve.h"

#include <solomon/libs/cpp/minidump/model/stack_trace.pb.h>

#include <contrib/libs/breakpad/src/client/linux/handler/exception_handler.h>
#include <contrib/libs/breakpad/src/google_breakpad/processor/minidump.h>
#include <contrib/libs/breakpad/src/google_breakpad/processor/minidump_processor.h>
#include <contrib/libs/breakpad/src/google_breakpad/processor/process_state.h>
#include <contrib/libs/breakpad/src/google_breakpad/processor/call_stack.h>
#include <contrib/libs/breakpad/src/google_breakpad/processor/stack_frame_cpu.h>
#include <contrib/libs/breakpad/src/processor/logging.h>

#include <build/scripts/c_templates/svnversion.h>

#include <util/string/builder.h>
#include <util/string/subst.h>
#include <util/stream/file.h>
#include <util/system/execpath.h>
#include <util/system/sysstat.h>

#include <fstream>

namespace NSolomon {
    void ParseThread(
            const google_breakpad::CallStack* stack,
            yandex::monitoring::minidump::Thread* thread,
            llvm::symbolize::LLVMSymbolizer& symbolizer)
    {
        int frameCount = stack->frames()->size();
        for (int frameIndex = 0; frameIndex < frameCount; ++frameIndex) {
            uint64_t address = stack->frames()->at(frameIndex)->ReturnAddress();
            yandex::monitoring::minidump::Frame* frame = thread->add_frames();

            auto lineInfo = Resolve(address, symbolizer);

            frame->set_function_name(std::string(lineInfo.FunctionName));
            SubstGlobal(lineInfo.FileName, GetArcadiaSourcePath(), "/-S");
            frame->set_path(std::string(lineInfo.FileName));
            frame->set_line(lineInfo.Line);
            frame->set_column(lineInfo.Column);
        }
    }

    void ParseStackTrace(
            const google_breakpad::ProcessState& processState,
            yandex::monitoring::minidump::StackTrace& stackTrace)
    {
        llvm::symbolize::LLVMSymbolizer symbolizer{llvm::symbolize::LLVMSymbolizer::Options()};
        int requesting_thread = processState.requesting_thread();
        if (requesting_thread != -1) {
            yandex::monitoring::minidump::Thread* thread = stackTrace.add_threads();
            thread->set_index(requesting_thread);
            thread->set_is_crashed(processState.crashed());
            if (processState.crashed()) {
                thread->set_exception_message(processState.exception_message());
            }
            ParseThread(processState.threads()->at(requesting_thread), thread, symbolizer);
        }
        int threadCount = processState.threads()->size();
        for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex) {
            if (requesting_thread != threadIndex) {
                yandex::monitoring::minidump::Thread* thread = stackTrace.add_threads();
                thread->set_index(threadIndex);
                ParseThread(processState.threads()->at(threadIndex), thread, symbolizer);
            }
        }
    }

    TString GetStackFilePath(const TString& path) {
        return path.substr(0, path.rfind('.')) + ".stack";
    }

    static bool DumpCallback(
            const google_breakpad::MinidumpDescriptor& descriptor,
            void*,
            bool succeeded)
    {
        Cout << "Dump path: " << descriptor.path() << Endl;
        Chmod(descriptor.path(), (S_IRUSR | S_IWUSR) | (S_IRGRP | S_IWGRP) | S_IROTH);
        google_breakpad::Minidump dump(descriptor.path());
        if (!dump.Read()) {
            Cerr << "Minidump " << dump.path() << " could not be read" << Endl;
            return !succeeded;
        }
        google_breakpad::MinidumpProcessor minidumpProcessor(nullptr, nullptr);

        // Increase the maximum number of threads and regions.
        google_breakpad::MinidumpThreadList::set_max_threads(std::numeric_limits<uint32_t>::max());
        google_breakpad::MinidumpMemoryList::set_max_regions(std::numeric_limits<uint32_t>::max());

        google_breakpad::ProcessState processState;
        if (minidumpProcessor.Process(&dump, &processState) != google_breakpad::PROCESS_OK) {
            Cerr << "MinidumpProcessor::Process failed" << Endl;
            return !succeeded;
        }
        TString stackPath = GetStackFilePath(descriptor.path());
        Cout << "Stack path: " << stackPath << Endl;
        std::ofstream fileOutput(stackPath.data(), std::ios::binary);
        yandex::monitoring::minidump::StackTrace stackTrace;
        ParseStackTrace(processState, stackTrace);
        Cout << "Stack trace size " << stackTrace.threads_size() << Endl;
        if (!stackTrace.SerializeToOstream(&fileOutput)) {
            Cerr << "Failed to write minidump." << Endl;
            return !succeeded;
        }
        return succeeded;
    }

    void DisableCoredump() {
        struct rlimit coreLimits;
        coreLimits.rlim_cur = 0;
        coreLimits.rlim_max = 0;
        setrlimit(RLIMIT_CORE, &coreLimits);
        Cout << "Coredumps are disabled" << Endl;
    }

    TString GetDefaultMinidumpPath() {
        TStringBuf dir, filename, path;
        path = GetExecPath();
        path.RSplit('/', dir, filename);
        return TStringBuilder{} << "/coredumps/" << filename;
    }

    void SetupMinidump(const yandex::monitoring::minidump::MinidumpConfig& minidumpConf) {
        if (minidumpConf.path()) {
            *GetMinidumpPath() = minidumpConf.path();
        } else {
            *GetMinidumpPath() = GetDefaultMinidumpPath();
        }
        Cerr << "Minidumps enabled in " << *GetMinidumpPath() << Endl;
        google_breakpad::LogStream::minimum_severity = google_breakpad::LogStream::SEVERITY_ERROR;
        google_breakpad::MinidumpDescriptor descriptor(*GetMinidumpPath());
        static google_breakpad::ExceptionHandler eh(
                descriptor,
                nullptr,
                DumpCallback,
                nullptr,
                true,
                -1,
                true);
        if (minidumpConf.disable_coredumps()) {
            DisableCoredump();
        }
    }
}
