#include "message.h"

#include <util/generic/yexception.h>
#include <util/stream/length.h>

namespace NSolomon {
    template <typename TProto>
    void TMessageBase<TProto>::ToStream(IOutputStream& os) const {
        if (Length() > 0) {
            os.Write(&Length_, TMessage::LENGTH_HEADER_SIZE);
            os.Write(Data_.data(), Length_);
        } else {
            TString serialized;
            Y_ENSURE(Message_->SerializeToString(&serialized), "cannot serialize message");
            Y_ENSURE(serialized.size() < TMessage::MAX_HEADER_SIZE);
            const ui32 size = static_cast<ui32>(serialized.size());
            os.Write(&size, sizeof size);
            os.Write(serialized.data(), serialized.size());
        }
    }

    IInputStream& operator>>(IInputStream& is, TMessage& value) {
        value.LoadFromStream(is);
        return is;
    }

    IOutputStream& operator<<(IOutputStream& os, const TMessage& message) {
        message.ToStream(os);
        return os;
    }

    IInputStream& operator>>(IInputStream& is, THeaderMessage& value) {
        value.LoadFromStream(is);
        return is;
    }

    IOutputStream& operator<<(IOutputStream& os, const THeaderMessage& message) {
        message.ToStream(os);
        return os;
    }

    template <typename TProto>
    TMessageBase<TProto>::TMessageBase(char* data, ui32 len)
        : Length_{len}
        , Data_{data, len}
    {
    }

    template <typename TProto>
    TMessageBase<TProto>::TMessageBase(TProto proto)
        : Message_{std::move(proto)}
    {
    }

    template <typename TProto>
    void TMessageBase<TProto>::LoadFromStream(IInputStream& is) {
        is.LoadOrFail(&Length_, TMessageBase::LENGTH_HEADER_SIZE);
        if (Length_ == 0) {
            return;
        }

        TLengthLimitedInput protoIn(&is, Length_);

        TProto proto;
        Y_ENSURE(proto.ParseFromArcadiaStream(&protoIn), "Message parsing failed");
        Message_ = std::move(proto);
    }

    TMessage::TMessage(TShardData sd)
        : TMessageBase<TShardData>{std::move(sd)}
    {
    }

    THeaderMessage::THeaderMessage(TMultiShardData::THeader header)
        : TMessageBase<TMultiShardData::THeader>(std::move(header))
    {
    }
} // namespace NSolomon
