#include "multi_shard.h"
#include "internal/message.h"
#include "internal/state.h"

#include <solomon/libs/cpp/multi_shard/proto/multi_shard.pb.h>

#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/text/text.h>

#include <util/stream/str.h>

namespace NSolomon::NMultiShard {
namespace {
    using namespace NMonitoring;

    using TEncoderFactory = std::function<IMetricEncoderPtr(IOutputStream*)>;

    TMultiShardData::THeader ToProtoHeader(THeader header) {
        TMultiShardData::THeader protoHeader;
        protoHeader.SetFormatVersion(static_cast<ui32>(header.FormatVersion));
        protoHeader.SetContinuationToken(std::move(header.ContinuationToken));
        return protoHeader;
    }

    class TMultiShardEncoder: public IMultiShardEncoder {
        EState State_{EState::Root};

        void SwitchForward(EState desired) {
            EState next{EState::MinValue};

            switch (State_) {
            case EState::Root:
                next = EState::Shard;
                break;
            case EState::Shard:
                break;
            case EState::Closed:
            case EState::MinValue:
            case EState::MaxValue:
                break;
            };

            Y_ENSURE(next != EState::MinValue && next != EState::MaxValue, "Cannot switch from " << State_ << " to " << desired);

            State_ = next;
        }

        void SwitchBack(EState desired) {
            EState next{EState::MinValue};

            switch (State_) {
            case EState::Root:
                next = EState::Closed;
                break;
            case EState::Shard:
                next = EState::Root;
                break;
            case EState::Closed:
            case EState::MinValue:
            case EState::MaxValue:
                break;
            };

            Y_ENSURE(next != EState::MinValue && next != EState::MaxValue, "Cannot switch from " << State_ << " to " << desired);

            State_ = next;
        }

    public:
        TMultiShardEncoder(TEncoderFactory factory, IOutputStream& os)
            : Factory_{std::move(factory)}
            , Out_{os}
        {
        }

        ~TMultiShardEncoder() {
            if (State_ != EState::Closed) {
                Close();
            }
        }

        void OnShardBegin(TString project, TString service, TString cluster) override {
            SwitchForward(EState::Shard);
            EnsureHeader();
            CurrentEncoder_ = Factory_(&Blob_);
            *ShardData_.MutableProject() = std::move(project);
            *ShardData_.MutableService() = std::move(service);
            *ShardData_.MutableCluster() = std::move(cluster);
        }

        void SetHeader(THeader header) override {
            WriteHeader(ToProtoHeader(header));
            HasHeader_ = true;
        }

        void OnShardEnd() override {
            SwitchBack(EState::Root);
            CurrentEncoder_->Close();
            *ShardData_.MutableData() = std::move(Blob_.Str());
            FlushOne();
            Reset();
        }

        void Close() override {
            SwitchBack(EState::Closed);
            Reset();
            Out_.Finish();
        }

        void OnStreamBegin() override {
            EnsureState();
            CurrentEncoder_->OnStreamBegin();
        }

        void OnStreamEnd() override {
            EnsureState();
            CurrentEncoder_->OnStreamEnd();
        }

        void OnCommonTime(TInstant time) override {
            EnsureState();
            CurrentEncoder_->OnCommonTime(time);
        }

        void OnMetricBegin(EMetricType kind) override {
            EnsureState();
            CurrentEncoder_->OnMetricBegin(kind);
        }

        void OnMetricEnd() override {
            EnsureState();
            CurrentEncoder_->OnMetricEnd();
        }

        void OnLabelsBegin() override {
            EnsureState();
            CurrentEncoder_->OnLabelsBegin();
        }

        void OnLabelsEnd() override {
            EnsureState();
            CurrentEncoder_->OnLabelsEnd();
        }

        void OnLabel(TStringBuf name, TStringBuf value) override {
            EnsureState();
            CurrentEncoder_->OnLabel(name, value);
        }

        void OnDouble(TInstant time, double value) override {
            EnsureState();
            CurrentEncoder_->OnDouble(time, value);
        }

        void OnInt64(TInstant time, i64 value) override {
            EnsureState();
            CurrentEncoder_->OnInt64(time, value);
        }

        void OnUint64(TInstant time, ui64 value) override {
            EnsureState();
            CurrentEncoder_->OnUint64(time, value);
        }

        void OnHistogram(TInstant time, IHistogramSnapshotPtr snapshot) override {
            EnsureState();
            CurrentEncoder_->OnHistogram(time, std::move(snapshot));
        }

        void OnSummaryDouble(TInstant time, ISummaryDoubleSnapshotPtr snapshot) override {
            EnsureState();
            CurrentEncoder_->OnSummaryDouble(time, std::move(snapshot));
        }

        void OnLogHistogram(TInstant time, TLogHistogramSnapshotPtr snapshot) override {
            EnsureState();
            CurrentEncoder_->OnLogHistogram(time, std::move(snapshot));
        }

    private:
        void EnsureState() const {
            if (State_ == EState::Closed) {
                ythrow yexception() << "Encoder has been already closed";
            } else if (State_ != EState::Shard || !CurrentEncoder_) {
                ythrow yexception() << "Shard encoder is not initialized. Did you forget to call OnShardBegin?";
            }
        }

        void EnsureHeader() const {
            Y_ENSURE(HasHeader_, "Header is not set");
        }

        void Reset() {
            if (CurrentEncoder_) {
                CurrentEncoder_.Reset();
            }

            Blob_.Clear();
        }

        void FlushOne() {
            TString serialized;
            TMessage msg{ShardData_};
            Out_ << msg;
            Out_.Flush();
        }

        void WriteHeader(TMultiShardData::THeader header) {
            THeaderMessage msg{std::move(header)};
            Out_ << msg;
            Out_.Flush();
        }

    private:
        TEncoderFactory Factory_;
        IOutputStream& Out_;

        TStringStream Blob_;
        TShardData ShardData_;
        IMetricEncoderPtr CurrentEncoder_;
        bool HasHeader_{false};
    };
} // namespace

    IMultiShardEncoderPtr CreateMultiShardEncoder(EFormat format, IOutputStream& os) {
        TEncoderFactory factory;
        switch (format) {
            case EFormat::SPACK:
                factory = [] (auto* os) {
                    return EncoderSpackV1(
                        os,
                        ETimePrecision::SECONDS,
                        ECompression::ZSTD,
                        EMetricsMergingMode::MERGE_METRICS
                    );
                };
                break;

            case EFormat::JSON:
                factory = [] (auto* os) {
                    return BufferedEncoderJson(os, 0);
                };
                break;

            case EFormat::TEXT:
                factory = [] (auto* os) {
                    return EncoderText(os, true);
                };
                break;

            default:
                ythrow yexception() << "Format is not supported";
        };

        return MakeHolder<TMultiShardEncoder>(std::move(factory), os);
    }
} // namespace NSolomon
