#include "multi_shard.h"

#include "internal/message.h"

#include <util/generic/algorithm.h>
#include <util/generic/buffer.h>
#include <util/stream/mem.h>
#include <util/system/unaligned_mem.h>

namespace NSolomon::NMultiShard {
namespace {
    constexpr auto MIN_VERSION = ui32(EFormatVersion::Min);
    constexpr auto MAX_VERSION = ui32(EFormatVersion::Max);

    THeader FromProtoHeader(const TMultiShardData::THeader& proto) {
        const auto version = proto.GetFormatVersion();
        Y_ENSURE((version > MIN_VERSION) && (version < MAX_VERSION),
            "Unknown format version: " << version);
        THeader header{
            proto.GetContinuationToken(),
            EFormatVersion(version),
        };

        return header;
    }
    enum class EState {
        NewMessageHeader = 0,
        NewMessage,
        NeedMore,
    };

    class TContinuousChunkDecoder: public IMultiShardContinuousChunkDecoder {
    public:
        explicit TContinuousChunkDecoder(IMessageHandler& handler)
            : Handler_{ handler }
        {
        }

    private:
        void Decode(TStringBuf buf) noexcept override {
            Buffer_ = buf;

            try {
                do {
                    if (CanReadLength()) {
                        ReadLength();
                    }

                    if (CanReadMessage()) {
                        if (!Read()) {
                            break;
                        }
                    }
                } while (CanReadLength());
            } catch (...) {
                Handler_.OnError(CurrentExceptionMessage());
                Handler_.OnStreamEnd();
                return;
            }

            Close();
        }

        bool Read() {
            switch (PrevState_) {
                case EState::NewMessageHeader: {
                    auto protoHeader = ReadHeader();
                    if (!FlushHeader(FromProtoHeader(std::move(protoHeader)))) {
                        Handler_.OnStreamEnd();
                        return false;
                    }

                    break;
                }

                case EState::NewMessage: {
                    auto msg = ReadMessage();
                    if (!FlushMessage(msg)) {
                        Handler_.OnStreamEnd();
                        return false;
                    }

                    break;
                }

                case EState::NeedMore:
                    ythrow yexception() << "Invalid state";
            }

            return true;
        }

        void Close() noexcept {
            if (Buffer_.Size() > 0) {
                Handler_.OnError("Some data left in buffer on close");
            }

            Handler_.OnStreamEnd();
        }

        bool CanReadLength() const {
            switch (State_) {
                case EState::NewMessageHeader:
                case EState::NewMessage:
                    return Buffer_.Size() >= TMessage::LENGTH_HEADER_SIZE;
                case EState::NeedMore:
                    break;
            }

            return false;
        }

        void ReadLength() {
            NextMessageSize_ = ReadUnaligned<decltype(NextMessageSize_)>(Buffer_.Data());
            Y_ENSURE(NextMessageSize_ > 0, "Message size cannot be zero");
            PrevState_ = State_;
            State_ = EState::NeedMore;
        }

        ui64 FullLength() const {
            return TMessage::LENGTH_HEADER_SIZE + NextMessageSize_;
        }

        bool CanReadMessage() const {
            return State_ == EState::NeedMore && Buffer_.Size() >= FullLength();
        }

        template <typename TMessageType>
        auto ReadProto() {
            TMemoryInput in{Buffer_.Data(), FullLength()};
            TMessageType msg;
            in >> msg;
            State_ = EState::NewMessage;
            return msg.Message();
        }

        TShardData ReadMessage() {
            return ReadProto<TMessage>();
        }

        TMultiShardData::THeader ReadHeader() {
            return ReadProto<THeaderMessage>();
        }

        bool FlushMessage(TShardData& shardData) {
            const auto ret = Handler_.OnShardData(
                std::move(*shardData.MutableProject()),
                std::move(*shardData.MutableCluster()),
                std::move(*shardData.MutableService()),
                std::move(*shardData.MutableData())
            );

            Buffer_.Skip(FullLength());

            NextMessageSize_ = 0;
            return ret;
        }

        bool FlushHeader(THeader&& header) {
            const auto ret = Handler_.OnHeader(std::move(header));

            Buffer_.Skip(FullLength());

            NextMessageSize_ = 0;
            return ret;
        }

    private:
        ui32 NextMessageSize_{0};
        IMessageHandler& Handler_;
        TStringBuf Buffer_;
        EState PrevState_{EState::NewMessageHeader};
        EState State_{EState::NewMessageHeader};
    };

} // namespace

THolder<IMultiShardContinuousChunkDecoder> CreateMultiShardContinuousChunkDecoder(IMessageHandler& handler) {
    return MakeHolder<TContinuousChunkDecoder>(handler);
}

} // namespace NSolomon
