#include "porto.h"

#include <contrib/libs/porto_api/libporto.hpp>

#include <util/string/cast.h>
#include <util/string/strip.h>
#include <util/string/split.h>

namespace NSolomon {

namespace {

const TString RUNNING = "running";

// parses list of form name: <N>; name: <N> ...
template <typename TFunc>
void ParseKeyValueList(TStringBuf sb, TFunc fn) {
    for (auto item: StringSplitter(sb).Split(';').SkipEmpty()) {
        const auto tok = item.Token();
        TStringBuf name, value;
        tok.Split(':', name, value);

        name = StripString(name);
        value = StripString(value);

        fn(name, value);
    }
}

ui64 ParseCpuMillis(const TString& s) {
     return ::FromString<ui64>(s) / 1e6;
}

ui64 ParseCpuLimit(TStringBuf s) {
    if (s.length() < 2 || s[s.length() - 1] != 'c') {
        ythrow yexception() << "Invalid cpu limit value: " << s;
    }

    return static_cast<ui64>(::FromString<double>(s.SubString(0, s.length() - 1)) * 1000);
}

ui64 ParseDiskIo(const TString& s) {
    TStringBuf sb{s};
    ui64 result{0};

    // fs: <N>; hw: <N>; ...
    ParseKeyValueList(sb, [&] (auto name, auto value) {
        if (name == TStringBuf("hw")) {
            result = ::FromString(value);
        }
    });

    return result;
}

ui64 ParseDiskIoMillis(const TString& s) {
    return ParseDiskIo(s) / 1e6;
}

using TNetValues = TVector<std::pair<TString, ui64>>;
TNetValues ParseNetStat(const TString& s) {
    TStringBuf sb{s};
    TNetValues result;

    ParseKeyValueList(sb, [&] (auto name, auto value) {
        result.emplace_back(name, ::FromString<ui64>(value));
    });

    return result;
}

TString GetRawProperty(Porto::Connection& conn, const TString& containerId, const TString& name) {
    int ret = 0;
    TString raw;
    if ((ret = conn.GetProperty(containerId, name, raw)) != 0) {
        int err;
        TString message;
        conn.GetLastError(err, message);
        ythrow yexception() << "Unable to get property " << name << ": "
            << err << " " << message;
    }

    return raw;
}

} // namespace

class TPortoStatReader : public IPortoStatClient {
public:
    using TProperty = ui64 TContainerStats::*;
    using TNetworkProperty = ui64 TContainerStats::TNetworkStats::*;

    class TStatBuilder {
    private:
        TContainerStats Stat_;
        Porto::Connection& Conn_;
        const TString& Id_;

    private:
        TString GetProperty(const TString& name) {
            return ::NSolomon::GetRawProperty(Conn_, Id_, name);
        }

    public:
        TStatBuilder(Porto::Connection& conn, const TString& containerId)
            : Conn_{conn}
            , Id_{containerId}
        {
        }

        template <typename TConv = ui64 (*)(const TString&)>
        TStatBuilder& GetProperty(const TString& name, TProperty prop, TConv converter = ::FromString<ui64>) {
            try {
                TString raw = GetProperty(name);
                (Stat_.*prop) = converter(raw);
            } catch (...) {
                // do nothing
            }

            return *this;
        }

        TStatBuilder& GetNetworkProperty(const TString& name, TNetworkProperty prop) {
            TString raw = GetProperty(name);

            for (auto&& ifstat: ParseNetStat(raw)) {
                (Stat_.Networks[ifstat.first].*prop) = ifstat.second;
            }

            return *this;
        }

        TContainerStats Build() {
            return Stat_;
        }
    };

    TVector<TString> Containers(const TString& filter) override {
        TVector<TString> result;
        int ret = 0;
        if ((ret = Conn_.List(result, filter)) != 0) {
            ythrow yexception() << "Unable to list containers: " << Conn_.GetLastError();
        }

        return result;
    }

    TVector<TString> RunningContainers(const TString& filter) override {
        auto containers = Containers(filter);

        const TVector<TString> vars = { "state" };
        TMap<TString, TMap<TString, Porto::GetResponse>> result;

        int ret;
        if ((ret = Conn_.Get(containers, vars, result)) != 0) {
            ythrow yexception() << "Unable to get containers state: " << Conn_.GetLastError();
        }

        using namespace std;
        auto newEnd = remove_if(begin(containers), end(containers), [&result] (const TString& name) {
            auto it = result.find(name);
            if (it == result.end()) {
                return true;
            }

            auto rsp = it->second.find("state");
            if (rsp == it->second.end()) {
                return true;
            }

            return rsp->second.Value != RUNNING;
        });

        containers.erase(newEnd, end(containers));
        return containers;
    }

    TVector<TVolumeStats> VolumeStats(const TString& container) override {
        TVector<Porto::Volume> volumes;
        if (Conn_.ListVolumes({}, container, volumes) != 0) {
            ythrow yexception() << "Unable to list volumes: " << Conn_.GetLastError();
        }

        TVector<TVolumeStats> stats;
        auto getProp = [] (auto& field, auto&& props, auto&& name) {
            ui64 value{0};
            if (auto* valueStr = props.FindPtr(name); valueStr && TryFromString(*valueStr, value)) {
                field = value;
            }
        };

        for (auto&& volume: volumes) {
            auto& stat = stats.emplace_back();
            stat.Path = volume.Path;
            getProp(stat.SpaceAvailableBytes, volume.Properties, "space_available");
            getProp(stat.SpaceUsedBytes, volume.Properties, "space_used");
            getProp(stat.SpaceLimitBytes, volume.Properties, "space_limit");
        }

        return stats;
    }

    TString RawProperty(const TString& containerId, const TString& name) override {
        return ::NSolomon::GetRawProperty(Conn_, containerId, name);
    }

    TContainerStats Stats(const TString& containerId) override {
        return TStatBuilder{Conn_, containerId}
            .GetProperty("cpu_usage", &TContainerStats::CpuUsageMillis, ParseCpuMillis)
            .GetProperty("cpu_usage_system", &TContainerStats::CpuUsageSystemMillis, ParseCpuMillis)
            .GetProperty("cpu_wait", &TContainerStats::CpuWaitMillis, ParseCpuMillis)
            .GetProperty("cpu_throttled", &TContainerStats::CpuThrottledMillis, ParseCpuMillis)
            .GetProperty("cpu_limit", &TContainerStats::CpuLimit, ParseCpuLimit)
            .GetProperty("cpu_guarantee", &TContainerStats::CpuGuarantee, ParseCpuLimit)
            .GetProperty("cpu_limit_total", &TContainerStats::CpuLimitTotal, ParseCpuLimit)
            .GetProperty("cpu_guarantee_total", &TContainerStats::CpuGuaranteeTotal, ParseCpuLimit)
            .GetProperty("memory_usage", &TContainerStats::MemoryUsageBytes)
            .GetProperty("anon_usage", &TContainerStats::AnonUsageBytes)
            .GetProperty("anon_max_usage", &TContainerStats::AnonMaxUsageBytes)
            .GetProperty("cache_usage", &TContainerStats::CacheUsageBytes)
            .GetProperty("mlock_usage", &TContainerStats::MLockUsageBytes)
            .GetProperty("memory_limit", &TContainerStats::MemoryLimit)
            .GetProperty("memory_limit_total", &TContainerStats::MemoryLimitTotal)
            .GetProperty("memory_guarantee", &TContainerStats::MemoryGuarantee)
            .GetProperty("memory_guarantee_total", &TContainerStats::MemoryGuaranteeTotal)
            .GetProperty("anon_limit", &TContainerStats::AnonLimit)
            .GetProperty("anon_limit_total", &TContainerStats::AnonLimitTotal)
            .GetProperty("io_read", &TContainerStats::DiskIoReadBytes, ParseDiskIo)
            .GetProperty("io_write", &TContainerStats::DiskIoWriteBytes, ParseDiskIo)
            .GetProperty("io_time", &TContainerStats::DiskIoTimeMillis, ParseDiskIoMillis)
            .GetProperty("major_faults", &TContainerStats::MajorPageFaults)
            .GetProperty("minor_faults", &TContainerStats::MinorPageFaults)
            .GetProperty("oom_kills", &TContainerStats::OomKills)
            .GetProperty("oom_kills_total", &TContainerStats::OomKillsTotal)
            .GetProperty("time", &TContainerStats::TimeSeconds)
            .GetNetworkProperty("net_tx_bytes", &TContainerStats::TNetworkStats::TxBytes)
            .GetNetworkProperty("net_rx_bytes", &TContainerStats::TNetworkStats::RxBytes)
            .GetNetworkProperty("net_tx_drops", &TContainerStats::TNetworkStats::TxDrop)
            .GetNetworkProperty("net_rx_drops", &TContainerStats::TNetworkStats::RxDrop)
            .GetNetworkProperty("net_tx_packets", &TContainerStats::TNetworkStats::TxPackets)
            .GetNetworkProperty("net_rx_packets", &TContainerStats::TNetworkStats::RxPackets)
            .Build();
    }

private:
    Porto::Connection Conn_;
};

IPortoStatClientPtr CreatePortoStatClient() {
    return new TPortoStatReader;
}

IOutputStream& operator<<(IOutputStream& os, const NSolomon::TContainerStats& stat) {
    os << "cpu_usage " << stat.CpuUsageMillis << "\n"
        << "cpu_usage_system " << stat.CpuUsageSystemMillis << "\n"
        << "cpu_wait " << stat.CpuWaitMillis << "\n"
        << "cpu_throttled " << stat.CpuThrottledMillis << "\n"
        << "cpu_limit " << stat.CpuLimit << "\n"
        << "cpu_guarantee " << stat.CpuGuarantee << "\n"
        << "cpu_limit_total " << stat.CpuLimitTotal << "\n"
        << "cpu_guarantee_total " << stat.CpuGuaranteeTotal << "\n"
        << "memory_usage " << stat.MemoryUsageBytes << "\n"
        << "anon_usage " << stat.AnonUsageBytes << "\n"
        << "anon_max_usage " << stat.AnonMaxUsageBytes << "\n"
        << "cache_usage " << stat.CacheUsageBytes << "\n"
        << "mlock_usage " << stat.MLockUsageBytes << "\n"
        << "memory_limit " << stat.MemoryLimit << "\n"
        << "memory_limit_total " << stat.MemoryLimitTotal << "\n"
        << "memory_guarantee " << stat.MemoryGuarantee << "\n"
        << "memory_guarantee_total " << stat.MemoryGuaranteeTotal << "\n"
        << "anon_limit " << stat.AnonLimit << "\n"
        << "anon_limit_total " << stat.AnonLimitTotal << "\n"
        << "time " << stat.TimeSeconds << "\n"
        << "major_faults " << stat.MajorPageFaults << "\n"
        << "minor_faults " << stat.MinorPageFaults << "\n"
        << "io_read " << stat.DiskIoReadBytes << "\n"
        << "io_write " << stat.DiskIoWriteBytes << "\n"
        << "io_time " << stat.DiskIoTimeMillis << "\n"
        ;

        for (auto&& net: stat.Networks) {
            os << net.first << ": net_rx_bytes " << net.second.RxBytes << "\n"
                << net.first << ": net_tx_bytes " << net.second.TxBytes << "\n"
                << net.first << ": net_tx_drops " << net.second.TxDrop << "\n"
                << net.first << ": net_rx_drops " << net.second.RxDrop << "\n"
                << net.first << ": net_tx_packets " << net.second.TxPackets << "\n"
                << net.first << ": net_rx_packets " << net.second.RxPackets << "\n"
                ;
        }

    return os;
}

IOutputStream& operator<<(IOutputStream& os, const NSolomon::TVolumeStats& stat) {
    os << "space_available " << stat.SpaceAvailableBytes << '\n'
        << "space_used " << stat.SpaceUsedBytes << '\n'
        << "space_guarantee " << stat.SpaceLimitBytes << '\n'
        ;

    return os;
}

} // namespace NSolomon
