#include "process_stats.h"

#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/system/rusage.h>

namespace NSolomon {
namespace {
    struct TLinuxStats: IProcessStatProvider {
        static constexpr auto SELF_STATUS_PATH{"/proc/self/status"};

        TResult GetSelfStats() noexcept override try {
            TFileInput status{SELF_STATUS_PATH};
            TString line;

            TProcessStats stats;

            auto extract = [](TStringBuf buf, auto& val) {
                TStringBuf ignore;
                StringSplitter(buf)
                    .SplitBySet(" \t")
                    .SkipEmpty()
                    .Take(2)
                    .CollectInto(&ignore, &val);
            };

            while (status.ReadLine(line)) {
                if (line.StartsWith("VmRSS")) {
                    extract(line, stats.MemRss);
                    stats.MemRss *= 1024;
                } else if (line.StartsWith("VmSwap")) {
                    extract(line, stats.MemSwap);
                    stats.MemSwap *= 1024;
                } else if (line.StartsWith("VmLib")) {
                    extract(line, stats.MemLib);
                    stats.MemLib *= 1024;
                } else if (line.StartsWith("Threads")) {
                    extract(line, stats.ThreadCount);
                }
            }

            auto rusage = TRusage::Get();
            stats.CpuUser = rusage.Utime.MilliSeconds();
            stats.CpuSystem = rusage.Stime.MilliSeconds();
            stats.MajorPageFaults = rusage.MajorPageFaults;

            return TResult::FromValue(stats);
        } catch (...) {
            return TResult::FromError(TStringBuilder{}
                    << "Error during process stats collection: "
                    << CurrentExceptionMessage());
        }
    };

    struct TNonLinuxStats: IProcessStatProvider {
        TResult GetSelfStats() noexcept override try {
            auto rusage = TRusage::Get();

            TProcessStats stats;
            stats.MemRss = TRusage::GetCurrentRSS();
            stats.CpuUser = rusage.Utime.MilliSeconds();
            stats.CpuSystem = rusage.Stime.MilliSeconds();
            stats.MajorPageFaults = rusage.MajorPageFaults;
            return TResult::FromValue(stats);
        } catch (...) {
            return TResult::FromError(TStringBuilder{}
                    << "Error during process stats collection: "
                    << CurrentExceptionMessage());
        }
    };
} // namespace

    IProcessStatProviderPtr CreateProcessStatProvider() {
#ifdef _linux_
        return MakeHolder<TLinuxStats>();
#else
        return MakeHolder<TNonLinuxStats>();
#endif
    }

} // namespace NSolomon
