#include "threads_stats.h"

#include <util/generic/strbuf.h>
#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/folder/path.h>

#include <ostream>

namespace NSolomon {
namespace {

const double TicksPerMicroSeconds = TicksPerMicrosec();
const i64 PageSize = sysconf(_SC_PAGESIZE);

} // namespace

NSolomon::TThreadStats ParseStatLine(const TString& statLine) {
    NSolomon::TThreadStats result;
    TVector<TStringBuf> statsVector;
    TStringBuf readLineLeft;
    TStringBuf readLineRight;
    TStringBuf pidStr;
    TStringBuf commStr;

    TStringBuf(statLine).Split(") ", readLineLeft, readLineRight);
    readLineLeft.Split(" (", pidStr, commStr);
    const TSetDelimiter<const char> delim(" ");
    Split(readLineRight, delim, statsVector);

    result.Pid = FromString(pidStr);
    result.Comm = commStr;
    result.State = statsVector[0][0];
    result.Ppid = FromString(statsVector[1]);
    result.Pgrp = FromString(statsVector[2]);
    result.Session = FromString(statsVector[3]);
    result.TtyNr = FromString(statsVector[4]);
    result.Tpgid = FromString(statsVector[5]);
    result.Flags = FromString(statsVector[6]);
    result.Minflt = FromString(statsVector[7]);
    result.Cminflt = FromString(statsVector[8]);
    result.Majflt = FromString(statsVector[9]);
    result.Cmajflt = FromString(statsVector[10]);
    result.Utime = static_cast<ui64>(FromString<ui32>(statsVector[11]) / TicksPerMicroSeconds);
    result.Stime = static_cast<ui64>(FromString<ui32>(statsVector[12]) / TicksPerMicroSeconds);
    result.Cutime = static_cast<i64>(FromString<i32>(statsVector[13]) / TicksPerMicroSeconds);
    result.Cstime = static_cast<i64>(FromString<i32>(statsVector[14]) / TicksPerMicroSeconds);
    result.Priority = FromString(statsVector[15]);
    result.Nice = FromString(statsVector[16]);
    result.NumThreads = FromString(statsVector[17]);
    result.Itrealvalue = FromString(statsVector[18]);
    result.Starttime = static_cast<ui64>(FromString<ui64>(statsVector[19]) / TicksPerMicroSeconds);
    result.Vsize = FromString(statsVector[20]);
    result.Rss = FromString<i64>(statsVector[21]) * PageSize;
    result.Rsslim = FromString(statsVector[22]);
    result.Startcode = FromString(statsVector[23]);
    result.Endcode = FromString(statsVector[24]);
    result.Startstack = FromString(statsVector[25]);
    result.Kstkesp = FromString(statsVector[26]);
    result.Kstkeip = FromString(statsVector[27]);
    result.Signal = FromString(statsVector[28]);
    result.Blocked = FromString(statsVector[29]);
    result.Sigignore = FromString(statsVector[30]);
    result.Sigcatch = FromString(statsVector[31]);
    result.Wchan = FromString(statsVector[32]);
    result.Nswap = FromString(statsVector[33]);
    result.Cnswap = FromString(statsVector[34]);
    result.ExitSignal = FromString(statsVector[35]);
    result.Processor = FromString(statsVector[36]);
    result.RtPriority = FromString(statsVector[37]);
    result.Policy = FromString(statsVector[38]);
    result.DelayacctBlkioTicks = FromString(statsVector[39]);
    result.GuestTime = FromString(statsVector[40]);
    result.CguestTime = FromString(statsVector[41]);
    result.StartData = FromString(statsVector[42]);
    result.EndData = FromString(statsVector[43]);
    result.StartBrk = FromString(statsVector[44]);
    result.ArgStart = FromString(statsVector[45]);
    result.ArgEnd = FromString(statsVector[46]);
    result.EnvStart = FromString(statsVector[47]);
    result.EnvEnd = FromString(statsVector[48]);
    result.ExitCode = FromString(statsVector[49]);

    return result;
}

namespace {

struct TLinuxStats: IThreadStatProvider {
    static constexpr auto SELF_TASK_PATH{"/proc/self/task"};

    TErrorOr<std::vector<ui64>, TGenericError> GetSelfThreads() noexcept override try {
        TVector<TString> threadIdsStr;
        {
            TFsPath taskPath{SELF_TASK_PATH};
            taskPath.ListNames(threadIdsStr);
        }

        std::vector<ui64> threadIds;
        for (const auto& idStr: threadIdsStr) {
            threadIds.push_back(FromString(idStr));
        }

        return TErrorOr<std::vector<ui64>, TGenericError>::FromValue(threadIds);
    } catch (...) {
        TString error = TStringBuilder{} << "Error during self threads collection: " << CurrentExceptionMessage();
        return TErrorOr<std::vector<ui64>, TGenericError>::FromError(error);
    }

    TErrorOr<TThreadStats, TGenericError> GetThreadStats(ui64 tid) noexcept override try {
        auto selfStatPath{"/proc/self/task/" + ToString(tid) + "/stat"};
        TFileInput stat{selfStatPath};

        TThreadStats result = ParseStatLine(stat.ReadLine());

        return TErrorOr<TThreadStats, TGenericError>::FromValue(result);
    } catch (...) {
        TString error = TStringBuilder{} << "Error during threads stats collection: " << CurrentExceptionMessage();
        return TErrorOr<TThreadStats, TGenericError>::FromError(error);
    }
};

} // namespace

std::ostream& operator<<(std::ostream& os, const TThreadStats& stats) {
    os << "Pid: " << stats.Pid << std::endl
       << "Comm: " << stats.Comm << std::endl
       << "State: " << stats.State << std::endl
       << "Ppid: " << stats.Ppid << std::endl
       << "Pgrp: " << stats.Pgrp << std::endl
       << "Session: " << stats.Session << std::endl
       << "TtyNr: " << stats.TtyNr << std::endl
       << "Tpgid: " << stats.Tpgid << std::endl
       << "Flags: " << stats.Flags << std::endl
       << "Minflt: " << stats.Minflt << std::endl
       << "Cminflt: " << stats.Cminflt << std::endl
       << "Majflt: " << stats.Majflt << std::endl
       << "Cmajflt: " << stats.Cmajflt << std::endl
       << "Utime: " << stats.Utime << std::endl
       << "Stime: " << stats.Stime << std::endl
       << "Cutime: " << stats.Cutime << std::endl
       << "Cstime: " << stats.Cstime << std::endl
       << "Priority: " << stats.Priority << std::endl
       << "Nice: " << stats.Nice << std::endl
       << "NumThreads: " << stats.NumThreads << std::endl
       << "Itrealvalue: " << stats.Itrealvalue << std::endl
       << "Starttime: " << stats.Starttime << std::endl
       << "Vsize: " << stats.Vsize << std::endl
       << "Rss: " << stats.Rss << std::endl
       << "Rsslim: " << stats.Rsslim << std::endl
       << "Startcode: " << stats.Startcode << std::endl
       << "Endcode: " << stats.Endcode << std::endl
       << "Startstack: " << stats.Startstack << std::endl
       << "Kstkesp: " << stats.Kstkesp << std::endl
       << "Kstkeip: " << stats.Kstkeip << std::endl
       << "Signal: " << stats.Signal << std::endl
       << "Blocked: " << stats.Blocked << std::endl
       << "Sigignore: " << stats.Sigignore << std::endl
       << "Sigcatch: " << stats.Sigcatch << std::endl
       << "Wchan: " << stats.Wchan << std::endl
       << "Nswap: " << stats.Nswap << std::endl
       << "Cnswap: " << stats.Cnswap << std::endl
       << "ExitSignal: " << stats.ExitSignal << std::endl
       << "Processor: " << stats.Processor << std::endl
       << "RtPriority: " << stats.RtPriority << std::endl
       << "Policy: " << stats.Policy << std::endl
       << "DelayacctBlkioTicks: " << stats.DelayacctBlkioTicks << std::endl
       << "GuestTime: " << stats.GuestTime << std::endl
       << "CguestTime: " << stats.CguestTime << std::endl
       << "StartData: " << stats.StartData << std::endl
       << "EndData: " << stats.EndData << std::endl
       << "StartBrk: " << stats.StartBrk << std::endl
       << "ArgStart: " << stats.ArgStart << std::endl
       << "ArgEnd: " << stats.ArgEnd << std::endl
       << "EnvStart: " << stats.EnvStart << std::endl
       << "EnvEnd: " << stats.EnvEnd << std::endl
       << "ExitCode: " << stats.ExitCode;

    return os;
}

std::unique_ptr<IThreadStatProvider> CreateThreadStatProvider() {
    return std::make_unique<TLinuxStats>();
}

} // namespace NSolomon
