#pragma once

#include <util/generic/ptr.h>
#include <util/generic/maybe.h>

#include <contrib/libs/croaring/cpp/roaring.hh>

namespace NSolomon::NSearch {

struct IBitmap: public TAtomicRefCount<IBitmap> {
    using TId = ui32;
    using TIdConsumer = std::function<void(TId)>;

    virtual ~IBitmap() = default;

    virtual size_t MemBytes() const noexcept = 0;
    /**
     * Return number of ids stored in current bitmap.
     */
    virtual size_t Size() const noexcept = 0;

    /**
     * Call provided function for all ids in the current bitmap.
     */
    virtual void Consume(const TIdConsumer& fn) const = 0;

    /**
     * Compute the union between the current bitmap and the provided bitmap,
     * writing the result in provided bitmap.
     */
    virtual void Or(roaring::Roaring& r) const noexcept = 0;

    /**
     * Compute the intersection between the current bitmap and the provided
     * bitmap, writing the result in the provided bitmap.
     */
    virtual void And(roaring::Roaring& r) const noexcept = 0;

    /**
     * Compute the difference between the current bitmap and the provided
     * bitmap, writing the result in the provided bitmap.
     */
    virtual void AndNot(roaring::Roaring& r) const noexcept = 0;

    /**
     * Convert the current bitmap to a roaring bitmap representation.
     */
    virtual roaring::Roaring ToRoaring() const noexcept = 0;

public:
    static TIntrusivePtr<IBitmap> Empty();
};

using IBitmapPtr = TIntrusivePtr<IBitmap>;

IBitmapPtr Optimize(const roaring::Roaring& r);
IBitmapPtr Optimize(const roaring::Roaring& r, ui32 max);
IBitmapPtr Or(const TVector<IBitmapPtr>& bitmaps);
IBitmapPtr And(const TVector<IBitmapPtr>& bitmaps);
IBitmapPtr Combine(
    const TVector<IBitmapPtr>& inclusive,
    const TVector<IBitmapPtr>& exclusive,
    ui32 max);

IBitmapPtr Invert(const IBitmap& bitmap, ui32 maxCount);

} // namespace NSolomon::NSearch
