#include "lsm_tree.h"
#include "bitmap_index.h"

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/json/writer/json.h>
#include <util/string/subst.h>

namespace NSolomon::NSearch {

namespace {

/**
 *  Writeable multi-segmented storage
 */
class TLsmTree final: public IStorage {
    using TSegments = TVector<IBitmapSegment::TPtr>;
private:
    TLsmLimits Limits_;

    TSegments Segments_; //TODO: use a versioned collection & multiple threads?
    TAtomicSharedPtr<NIntern::TStringPool> Intern_;
    THolder<ISearchIndexBuilder> Builder_;
    roaring::Roaring Removals_;
public:
    TLsmTree(const TLsmLimits& limits)
        : Limits_(limits)
        , Intern_(MakeAtomicShared<NIntern::TStringPool>())
    {
    }

    ISearchIndexPtr Finalize() override {
        return this;
    }

    void Add(const TVector<std::pair<NMonitoring::ILabelsPtr, ui32>>& data) override {
        if (!Builder_) {
            Builder_ = CreateBitmapIndexBuilder(Intern_);
        }
        Builder_->Add(data);
    }

    void MarkRemoved(const TVector<ui32>& metricIds) override {
        Removals_.addMany(metricIds.size(), metricIds.data());
        // Builder->MarkRemoved is not called
    }

    void Flush() override
    {
        IBitmapPtr removals = Removals_.isEmpty() ? IBitmapPtr{} : ::NSolomon::NSearch::Optimize(Removals_);
        if (Builder_) {
            ISearchIndex::TPtr si = Builder_->Finalize();
            IBitmapSegment::TPtr bs = CastToSegment(si);
            Y_ASSERT(bs);
            if (bs) {
                AddSegment(std::move(bs), removals);
            }
            Builder_.Destroy();
        } else if (removals) {
            ApplyRemovals(*removals);
        }
        Removals_ = {};
    }

    ISearchResult::TPtr Search(const TSelectors& selectors, ui32 max) const override {
        TVector<ISearchResult::TPtr> parts;
        parts.reserve(Segments_.size());
        for (const auto& segment: Segments_) {
            parts.push_back(segment->Search(selectors, max));
        }
        return CombineSegmentResults(parts, max);
    }

    bool NeedsOptimize() const override {
        return true; //TODO: implement;
    }
    TString DebugString(TString key) const override {
        using namespace NJson;
        TJsonValue v(JSON_NULL);
        if (key == "segments") {
            v = TJsonValue(JSON_ARRAY);
            for (const auto& seg: Segments_) {
                TJsonValue s(JSON_MAP);
                s.InsertValue("size", TJsonValue(seg->Size()));
                v.AppendValue(s);
            }
        }
        return FormatDebugString(v);
    }

    size_t AllocatedBytes() const override {
        size_t sizeBytes = sizeof(*this)
                + Segments_.capacity() * sizeof(IBitmapSegment::TPtr)
                + Intern_->AllocatedBytes()
                + Removals_.getSizeInBytes();

        for (const auto& segment: Segments_) {
            if (segment) {
                sizeBytes += segment->AllocatedBytes();
            }
        }

        return sizeBytes;
    }

private:
    void ApplyRemovals(const IBitmap& removals) {
        for (auto& segment: Segments_) {
            segment->Remove(removals);
        }
    }

    void AddSegment(IBitmapSegment::TPtr&& bs, const IBitmapPtr& removals) {
        // Mark removals
        if (removals) {
            ApplyRemovals(*removals);
            bs->Remove(*removals);
        }

        // Mark replacements
        for (auto& segment: Segments_) {
            segment->Remove(*bs->GetAllMetrics());
        }

        // Insert sorted
        size_t sz = bs->Size();
        auto i = Segments_.begin();
        for (; i != Segments_.end(); ++i) {
            if ((*i)->Size() <= sz)
                break;
        }
        Segments_.insert(i, bs);

        //TODO: check if Optimize is needed
    }

    template <typename TIter>
    static std::pair<TIter, TIter> GetSegmentsForMerge(TIter begin, TIter end, size_t min, size_t max) {
        while (begin != end && (*begin)->Size() >= max) {
            ++begin;
        }
        auto p = begin;
        while (p != end && (*p)->Size() >= min) {
            ++p;
        }
        return {begin, p};
    }

    template <typename TIter>
    static IBitmapSegment::TPtr DoMergeSegments(TIter begin, TIter end) {
         TVector<IBitmapSegment::TPtr> segments;
         segments.insert(segments.end(), begin, end);
         return MergeSegments(segments);
    }

public:
    /**
     *  @brief Merge internal data structures of the segments
     */
    bool Optimize() override {
        TSegments newSegments;

        auto nextStep = [min = Limits_.MaxSegmentSize](size_t sizeRBegin) {
            size_t sizeREnd = sizeRBegin / 2;
            if (sizeREnd <= min) {
                sizeREnd = 0; // include all the segments smaller than MinSegmentSize in the last bucket
            }
            return sizeREnd;
        };

        size_t sizeRBegin = Limits_.MaxSegmentSize;
        auto pPrev = Segments_.rbegin();
        while (sizeRBegin) {
            size_t sizeREnd = nextStep(sizeRBegin);
            auto&& [p0, p1] = GetSegmentsForMerge(pPrev, Segments_.rend(), sizeREnd, sizeRBegin);
            if (pPrev == Segments_.rbegin() && p0 != pPrev) {
                // copy segments that is bigger that MaxSegmentSize "as is"
                newSegments.insert(newSegments.end(), Segments_.rbegin(), p0);
            }

            if (p0 == Segments_.rend()) {
                break;
            }

            size_t cnt = std::distance(p0, p1);
            if (cnt == 1) {
                newSegments.push_back(*p0++);
            } else {
                newSegments.push_back(DoMergeSegments(p0, p1));
            }

            sizeRBegin = sizeREnd;
            pPrev = p1;
        }

        Segments_.swap(newSegments);
        return false;
    }


private:
    static TString FormatDebugString(const NJson::TJsonValue& v) {
        NJsonWriter::TBuf w;
        w.WriteJsonValue(&v, /*sortKeys=*/true);
        TString s = w.Str();
        SubstGlobal(s, "\"", "'");
        return s;
    }
};

}

TIntrusivePtr<IStorage> CreateLsmTree(const TLsmLimits& limits) {
    return MakeIntrusive<TLsmTree>(limits);
};

} // namespace NSolomon::NSearch
