#include <solomon/libs/cpp/search/bitmap_index.h>
#include <solomon/libs/cpp/search/lsm_tree.h>

#include <solomon/libs/cpp/selectors/selectors.h>
#include <library/cpp/testing/gtest/gtest.h>

using NMonitoring::TLabels;
using namespace NSolomon;
using namespace NSolomon::NSearch;

namespace {
class SearchIndexTest : public ::testing::Test {
public:
    static ISearchIndexPtr MakeSearch(const TVector<NMonitoring::TLabels>& metrics) {
        auto builder = CreateBitmapIndexBuilder();
        ui32 metricId = 0;

        TVector<std::pair<NMonitoring::ILabelsPtr, ui32>> toAdd;
        for (const auto& item: metrics) {
            NMonitoring::ILabelsPtr labels{new TLabels(item)};
            toAdd.emplace_back(std::move(labels), ++metricId);
        }

        builder->Add(toAdd);
        return builder->Finalize();

    }

    static TVector<ui32> ToVector(const ISearchResultPtr& result) {
        TVector<ui32> ids;
        ui32 id;
        while ((id = result->NextId()) != ISearchResult::npos) {
            ids.push_back(id);
        }
        return ids;
    }
};

static TVector<std::pair<NMonitoring::ILabelsPtr, ui32>> ConvertLabels(const TVector<std::pair<NMonitoring::TLabels, ui32>>& metrics) {
    TVector<std::pair<NMonitoring::ILabelsPtr, ui32>> toAdd;
    for (const auto& [l, i]: metrics) {
        NMonitoring::ILabelsPtr labels{new TLabels(l)};
        toAdd.emplace_back(std::move(labels), i);
    }
    return toAdd;
}

}

TEST_F(SearchIndexTest, BitmapBasics) {
    auto index = MakeSearch({
        {{"bb", "cc"}, {"dd", "ee"}}
    });

    {
        TSelectors query;
        query.Add("aa", "-", false);
        auto result = index->Search(query, 100);

        EXPECT_EQ(ToVector(result), TVector<ui32>{1});
    }
    {
        TSelectors query;
        query.Add("bb", "ee", false);
        auto result = index->Search(query, 100);

        EXPECT_EQ(ToVector(result), TVector<ui32>{});
        EXPECT_EQ(result->ToVector(), TVector<ui32>{});
    }
    {
        TSelectors query;
        query.Add("bb", "cc", false);
        auto result = index->Search(query, 100);

        EXPECT_EQ(ToVector(result), TVector<ui32>{1});
        EXPECT_EQ(result->ToVector(), TVector<ui32>{1});
    }
}

TEST_F(SearchIndexTest, EmptySelector) {
    auto index = MakeSearch({
        {{"host", "aaa"}, {"sensor", "xxx"}},
        {{"host", "aaa"}, {"sensor", "yyy"}},
        {{"host", "bbb"}, {"sensor", "xxx"}},
        {{"host", "bbb"}, {"sensor", "yyy"}}
    });

    {
        TSelectors query;
        auto result = index->Search(query, 100);
        EXPECT_EQ(ToVector(result), TVector<ui32>({1, 2, 3, 4}));
    }
}

TEST_F(SearchIndexTest, NegativeSelector) {
    auto index = MakeSearch({
        {{"host", "aaa"}, {"sensor", "xxx"}},
        {{"host", "aaa"}, {"sensor", "yyy"}},
        {{"host", "bbb"}, {"sensor", "xxx"}},
        {{"host", "bbb"}, {"sensor", "yyy"}}
    });

    {
        TSelectors query;
        query.Add("host", "bbb", true);
        auto result = index->Search(query, 100);
        EXPECT_EQ(ToVector(result), TVector<ui32>({1, 2}));
    }
}

TEST_F(SearchIndexTest, SomeLabelsIsMissing) {
    auto index = MakeSearch({
        {{"host", "aaa"}, {"sensor", "xxx"}},
        {{"host", "bbb"}, {"sensor", "xxx"}},
        {{"host", "aaa"}, {"sensor", "yyy"}, {"user", "nnn"}},
        {{"host", "bbb"}, {"sensor", "yyy"}, {"user", "mmm"}}
    });

    EXPECT_EQ(TVector<ui32>({3}), index->Search(ParseSelectors("{user=nnn}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({4}), index->Search(ParseSelectors("{user!=nnn}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({3,4}), index->Search(ParseSelectors("{user=*}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{user=-}"), 100)->ToVector());
}

TEST_F(SearchIndexTest, NonExistentLabel) {
    auto index = MakeSearch({
        {{"aa", "bb"}, {"cc", "dd"}},
        {{"aa", "cc"}, {"bb", "dd"}}
    });

    //exclude nonexistent label
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=value|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=*|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=-|*}"), 100)->ToVector());

    // non existent with negative
    EXPECT_EQ(TVector<ui32>(), index->Search(ParseSelectors("{xx!=value}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>(), index->Search(ParseSelectors("{xx!=value, aa=bb}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>(), index->Search(ParseSelectors("{xx!=value|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>(), index->Search(ParseSelectors("{xx!=-}"), 100)->ToVector());

    // existentAndExcludeNonexistentLabel
    EXPECT_EQ(TVector<ui32>({1}), index->Search(ParseSelectors("{aa=bb, xx=-}"), 100)->ToVector());
}

TEST_F(SearchIndexTest, MultiGlobWithAbsent) {
    auto index = MakeSearch({
        {{"aa", "bb"}, {"cc", "dd"}},
        {{"aa", "cc"}, {"bb", "dd"}},
        {{"bb", "dd"}},
        {{"dd", "ee"}}
    });

    EXPECT_EQ(TVector<ui32>({1,3,4}), index->Search(ParseSelectors("{aa=bb|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,3,4}), index->Search(ParseSelectors("{aa=b*|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({2,3,4}), index->Search(ParseSelectors("{aa=*c|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2,3,4}), index->Search(ParseSelectors("{aa=*|-}"), 100)->ToVector());
}

TEST_F(SearchIndexTest, Substrings) {
    auto index = MakeSearch({
        {{"host", "aaa001"}, {"sensor", "xxxnnn"}},
        {{"host", "aaa002"}, {"sensor", "yyymmm"}},
        {{"host", "bbb001"}, {"sensor", "xxxmmm"}},
        {{"host", "bbb002"}, {"sensor", "yyynnn"}}
    });

    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{host=aaa001|aaa002}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{host=aaa*}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{host=aaa*|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2,3,4}), index->Search(ParseSelectors("{host=aaa*|*}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,4}), index->Search(ParseSelectors("{sensor=*nnn}"), 100)->ToVector());
}

TEST_F(SearchIndexTest, Limit) {
    auto index = MakeSearch({
        {{"aa", "bb"}, {"cc", "dd"}},
        {{"aa", "cc"}, {"bb", "dd"}},
        {{"aa", "dd"}, {"bb", "cc"}}
    });
    EXPECT_EQ(1u, index->Search(ParseSelectors("{aa=*}"), 1)->ToVector().size());
    EXPECT_EQ(2u, index->Search(ParseSelectors("{aa=*}"), 2)->Size());
    EXPECT_EQ(2u, ToVector(index->Search(ParseSelectors("{aa=*}"), 2)).size());
}

TEST_F(SearchIndexTest, EmptyResults) {
    auto index = MakeSearch({
        {{"a", "b"}, {"c", "d"}},
        {{"a", "f"}, {"c", "g"}}
    });
    // non existent label name
    EXPECT_EQ(0u, index->Search(ParseSelectors("{x=y}"), 100)->Size());
    // non existent label value
    EXPECT_EQ(0u, index->Search(ParseSelectors("{a=y}"), 100)->Size());
    // zero intersection
    EXPECT_EQ(0u, index->Search(ParseSelectors("{a=b, c=g}"), 100)->Size());
    EXPECT_EQ(0u, index->Search(ParseSelectors("{a=b, c!=d}"), 100)->Size());
}

TEST_F(SearchIndexTest, LabelInAlmostAllDocs) {
    auto index = MakeSearch({
        {{"a", "10"}, {"b", "x"}},
        {{"a", "10"}, {"b", "y"}},
        {{"a", "10"}, {"b", "z"}},
        {{"a", "20"}, {"b", "w"}}
    });
    EXPECT_EQ(TVector<ui32>({1,2,3}), index->Search(ParseSelectors("{a=10}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2,3}), index->Search(ParseSelectors("{a!=20}"), 100)->ToVector());
}

TEST_F(SearchIndexTest, IsNotAny) {
    // extra specifications from yrum@
    auto index = MakeSearch({
        {{"aa", "bb"}, {"cc", "dd"}},
        {{"aa", "cc"}, {"bb", "dd"}}
    });

    EXPECT_EQ(TVector<ui32>({1}), index->Search(ParseSelectors("{xx!=*, aa=bb}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>(), index->Search(ParseSelectors("{xx!=y*, aa=bb}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>(), index->Search(ParseSelectors("{xx!=*|-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=-}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=-|y*}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({1,2}), index->Search(ParseSelectors("{xx=-|*}"), 100)->ToVector());
}


//
// LSM-based index
//

TEST(LsmStorageTest, Basics) {
    TIntrusivePtr<IStorage> storage = CreateLsmTree(TLsmLimits());

    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd1{
        {{{"host", "aaa"}, {"sensor", "xxx"}}, 5u},
        {{{"host", "aaa"}, {"sensor", "yyy"}}, 8u},
    };
    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd2{
        {{{"host", "bbb"}, {"sensor", "xxx"}}, 6u},
        {{{"host", "bbb"}, {"sensor", "yyy"}}, 9u},
    };

    storage->Add(ConvertLabels(toAdd1));
    {
        TSelectors query;
        auto result = storage->Search(query, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({}));
    }
    storage->Flush();
    {
        TSelectors query1;
        auto result = storage->Search(query1, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({5, 8}));

        TSelectors query2;
        query2.Add("sensor", "yyy");
        result = storage->Search(query2, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({8}));
    }
    storage->Add(ConvertLabels(toAdd2));
    storage->Flush();
    {
        TSelectors query1;
        auto result = storage->Search(query1, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({5, 6, 8, 9}));

        TSelectors query2;
        query2.Add("sensor", "yyy");
        result = storage->Search(query2, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({8, 9}));

        TSelectors query3;
        query3.Add("host", "aaa");
        result = storage->Search(query3, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({5, 8}));
    }

    storage->Optimize();
    {
        TSelectors query1;
        auto result = storage->Search(query1, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({5, 6, 8, 9}));

        TSelectors query2;
        query2.Add("sensor", "yyy");
        result = storage->Search(query2, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({8, 9}));

        TSelectors query3;
        query3.Add("host", "aaa");
        result = storage->Search(query3, 100);
        EXPECT_EQ(result->ToVector(), TVector<ui32>({5, 8}));
    }

}


TEST(LsmStorageTest, SegmentSizeBasics) {
    TIntrusivePtr<IStorage> storage = CreateLsmTree(TLsmLimits());

    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd1{
        {{{"host", "aaa"}, {"sensor", "xxx"}}, 5u},
        {{{"host", "aaa"}, {"sensor", "yyy"}}, 8u},
    };
    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd2{
        {{{"host", "bbb"}, {"sensor", "xxx"}}, 6u},
        {{{"host", "bbb"}, {"sensor", "yyy"}}, 9u},
        {{{"host", "bbb"}, {"sensor", "zzz"}}, 10u},
    };


    EXPECT_EQ("[]", storage->DebugString("segments"));
    storage->Add(ConvertLabels(toAdd1));
    storage->Flush();
    EXPECT_EQ("[{'size':2}]", storage->DebugString("segments"));

    storage->Optimize(); // does nothing
    EXPECT_EQ("[{'size':2}]", storage->DebugString("segments"));

    storage->Add(ConvertLabels(toAdd2));
    storage->Flush(); // Adds a second (bigger) segment
    EXPECT_EQ("[{'size':3},{'size':2}]", storage->DebugString("segments"));

    EXPECT_EQ(true, storage->NeedsOptimize());
    storage->Optimize(); // merges the segments together
    EXPECT_EQ("[{'size':5}]", storage->DebugString("segments"));
}


TEST(LsmStorageTest, Replacement) {
    TIntrusivePtr<IStorage> storage = CreateLsmTree(TLsmLimits());

    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd1{
        {{{"host", "aaa"}, {"sensor", "xxx"}}, 1u},
        {{{"host", "bbb"}, {"sensor", "zzz"}}, 2u},
    };
    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd2{
        {{{"host", "bbb"}, {"sensor", "xxx"}}, 1u},
        {{{"host", "aaa"}, {"sensor", "zzz"}}, 2u},
    };

    storage->Add(ConvertLabels(toAdd1));
    storage->Flush();
    {
        EXPECT_EQ(TVector<ui32>({1,2}), storage->Search(ParseSelectors("{host=*}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({1}), storage->Search(ParseSelectors("{host=aaa}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({}), storage->Search(ParseSelectors("{host=bbb, sensor=xxx}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=bbb, sensor=zzz}"), 100)->ToVector());
    }
    storage->Add(ConvertLabels(toAdd2));
    storage->Flush();
    {
        EXPECT_EQ(TVector<ui32>({1,2}), storage->Search(ParseSelectors("{host=*}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=aaa}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({1}), storage->Search(ParseSelectors("{host=bbb, sensor=xxx}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({}), storage->Search(ParseSelectors("{host=bbb, sensor=zzz}"), 100)->ToVector());
    }
    storage->Optimize(); // merge two segments together
    EXPECT_EQ("[{'size':2}]", storage->DebugString("segments"));
    {
        EXPECT_EQ(TVector<ui32>({1,2}), storage->Search(ParseSelectors("{host=*}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=aaa}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({1}), storage->Search(ParseSelectors("{host=bbb, sensor=xxx}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=aaa, sensor=zzz}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({}), storage->Search(ParseSelectors("{host=aaa, sensor=xxx}"), 100)->ToVector());
        EXPECT_EQ(TVector<ui32>({}), storage->Search(ParseSelectors("{host=bbb, sensor=zzz}"), 100)->ToVector());
    }
}

TEST(LsmStorageTest, Removal) {
    TIntrusivePtr<IStorage> storage = CreateLsmTree(TLsmLimits());

    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd1{
        {{{"host", "aaa"}, {"sensor", "xxx"}}, 1u},
        {{{"host", "bbb"}, {"sensor", "zzz"}}, 2u},
    };
    TVector<std::pair<NMonitoring::TLabels, ui32>> toAdd2{
        {{{"host", "bbb"}, {"sensor", "xxx"}}, 1u},
        {{{"host", "aaa"}, {"sensor", "zzz"}}, 2u},
    };
    storage->Add(ConvertLabels(toAdd1));
    storage->Flush();
    storage->Add(ConvertLabels(toAdd2));
    storage->MarkRemoved({1u});
    storage->Flush();
    EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=*}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=aaa}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({}), storage->Search(ParseSelectors("{host=bbb}"), 100)->ToVector());
    storage->Optimize(); // merge two segments together
    EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=*}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({2}), storage->Search(ParseSelectors("{host=aaa}"), 100)->ToVector());
    EXPECT_EQ(TVector<ui32>({}), storage->Search(ParseSelectors("{host=bbb}"), 100)->ToVector());

    storage->MarkRemoved({2u});
    storage->Flush();
    EXPECT_EQ(TVector<ui32>(), storage->Search(TSelectors{}, 100)->ToVector());
    storage->Optimize();
    EXPECT_EQ(TVector<ui32>(), storage->Search(TSelectors{}, 100)->ToVector());
}
