#include "secrets.h"

#include <solomon/protos/secrets/secrets_config.pb.h>

#include <google/protobuf/messagext.h>
#include <google/protobuf/text_format.h>

#include <util/generic/hash.h>
#include <util/stream/file.h>

#include <utility>

namespace NSolomon::NSecrets {
namespace {

const TStringBuf YAV_SCHEME = "yav://";

class TEmptySecretProvider: public ISecretProvider {
public:
    std::optional<TString> GetSecret(TStringBuf /*key*/) const override {
        return std::nullopt;
    }
};

class TInlineSecretProvider: public ISecretProvider {
public:
    std::optional<TString> GetSecret(TStringBuf key) const override {
        return std::make_optional<TString>(key);
    }
};

class TMapSecretProvider: public ISecretProvider {
public:
    explicit TMapSecretProvider(THashMap<TString, TString> secrets)
        : Secrets_{std::move(secrets)}
    {
    }

    std::optional<TString> GetSecret(TStringBuf key) const override {
        if (key.starts_with(YAV_SCHEME)) {
            key.remove_prefix(YAV_SCHEME.size());
        }
        auto it = Secrets_.find(key);
        if (it == Secrets_.end()) {
            return std::nullopt;
        }
        return {it->second};
    }

private:
    THashMap<TString, TString> Secrets_;
};

THashMap<TString, TString> ParseSecrets(TStringBuf content) {
    yandex::solomon::config::SecretsConfig config;

    google::protobuf::io::ArrayInputStream in(content.Data(), content.Size());
    Y_ENSURE(google::protobuf::TextFormat::Parse(&in, &config), "cannot parse secrets config");

    THashMap<TString, TString> secrets;
    for (int i = 0; i < config.secrets_size(); ++i) {
        const auto& secret = config.secrets(i);
        bool emplaced = secrets.emplace(secret.key(), secret.value()).second;
        Y_ENSURE(emplaced, "duplicate secret key: " << secret.key());
    }

    return secrets;
}

} // namespace

std::shared_ptr<ISecretProvider> EmptySecretProvider() {
    return std::make_shared<TEmptySecretProvider>();
}

std::shared_ptr<ISecretProvider> InlineSecretProvider() {
    return std::make_shared<TInlineSecretProvider>();
}

std::shared_ptr<ISecretProvider> FileSecretProvider(const TString& path) {
    TString content = TFileInput(path).ReadAll();
    return std::make_shared<TMapSecretProvider>(ParseSecrets(content));
}

std::shared_ptr<ISecretProvider> StringSecretProvider(TStringBuf content) {
    return std::make_shared<TMapSecretProvider>(ParseSecrets(content));
}

} // namespace NSolomon::NSecrets
