#pragma once

#include <util/generic/string.h>
#include <util/generic/strbuf.h>

#include <optional>

namespace NSolomon::NSecrets {

/**
 * Interface of secrets provider.
 */
class ISecretProvider {
public:
    virtual ~ISecretProvider() = default;

    /**
     * Retrieve secret from storage by unique key.
     *
     * @param key   secret key
     * @return      secret value, non empty if secret present
     */
    virtual std::optional<TString> GetSecret(TStringBuf key) const = 0;
};

/**
 * Create secret provider which always returns empty values.
 *
 * @return new instance of secret provider
 */
std::shared_ptr<ISecretProvider> EmptySecretProvider();

/**
 * Create secret provider which returns secret directly inlined in configuration file.
 *
 * @return new instance of secret provider
 */
std::shared_ptr<ISecretProvider> InlineSecretProvider();

/**
 * Create secret provider which has secrets loaded from given file.
 *
 * @param path   path of the file with secrets
 * @return new instance of secret provider
 */
std::shared_ptr<ISecretProvider> FileSecretProvider(const TString& path);

/**
 * Create secret provider which has secrets loaded from given file.
 *
 * @param content   secrets content
 * @return new instance of secret provider
 */
std::shared_ptr<ISecretProvider> StringSecretProvider(TStringBuf content);

} // namespace NSolomon::NSecrets
