#include "matchers.h"

#include <solomon/libs/cpp/glob/glob.h>

#include <contrib/libs/re2/re2/re2.h>

#include <util/generic/yexception.h>
#include <util/memory/addstorage.h>
#include <util/string/split.h>

namespace NSolomon {
namespace {

constexpr TStringBuf ANY_PATTERN = "*";
constexpr TStringBuf ABSENT_PATTERN = "-";

/**
 * Helper class to reduce number of allocations and minimize footprint of matchers.
 * It places string pattern in contiguous block of memory right after matcher object.
 */
template <typename TDerived>
class TMatcherWithString: public TAdditionalStorage<TDerived> {
public:
    TMatcherWithString(TStringBuf string) noexcept {
        std::memcpy(this->AdditionalData(), string.data(), string.size());
    }

    TStringBuf GetString() const noexcept {
        return { reinterpret_cast<const char*>(this->AdditionalData()), this->AdditionalDataLength() };
    }
};

/**
 * Memory footprint:
 *
 * +--------+--------+--------+~~~~~~~~~~~~~~~~~~~+
 * |  Vptr  | RefCnt | Length |      pattern      |
 * +--------+--------+---+----+~~~~~~~~~~~~~~~~~~~+
 *                       |     \_________________/
 *                        \____________/
 */
class TExactMatcher final: public IMatcher, public TMatcherWithString<TExactMatcher> {
public:
    TExactMatcher(TStringBuf pattern) noexcept
        : TMatcherWithString<TExactMatcher>(pattern)
    {
    }

    EMatcherType Type() const noexcept override {
        return EMatcherType::EXACT;
    }

    TStringBuf Pattern() const noexcept override {
        return GetString();
    }

    bool Match(TStringBuf value) const override {
        return GetString() == value;
    }

    bool MatchesAbsent() const override {
        return false;
    }

    void Print(bool negative, IOutputStream& os) const override {
        os << (negative ? TStringBuf{"!== "} : TStringBuf{"== "});
        os << '\'' << GetString() << '\'';
    }
};
static_assert (sizeof(TExactMatcher) == 16, "");

class TGlobMatcher final: public IMatcher, public TMatcherWithString<TGlobMatcher> {
public:
    TGlobMatcher(TStringBuf pattern) noexcept
        : TMatcherWithString<TGlobMatcher>(pattern)
    {
    }

    EMatcherType Type() const noexcept override {
        return EMatcherType::GLOB;
    }

    TStringBuf Pattern() const noexcept override {
        return GetString();
    }

    bool Match(TStringBuf value) const override {
        return NSolomon::IsGlobMatch(GetString(), value);
    }

    bool MatchesAbsent() const override {
        return false;
    }

    void Print(bool negative, IOutputStream& os) const override {
        os << (negative ? TStringBuf{"!= "} : TStringBuf{"= "});
        os << '\'' << GetString() << '\'';
    }
};

class TRegexMatcher final: public IMatcher, public TMatcherWithString<TRegexMatcher> {
public:
    TRegexMatcher(TStringBuf pattern)
        : TMatcherWithString<TRegexMatcher>(pattern)
        , Regexp_(re2::StringPiece(pattern), Options())
    {
    }

    EMatcherType Type() const noexcept override {
        return EMatcherType::REGEX;
    }

    TStringBuf Pattern() const noexcept override {
        return GetString();
    }

    bool Match(TStringBuf value) const override {
        return RE2::FullMatch(re2::StringPiece(value), Regexp_);
    }

    bool MatchesAbsent() const override {
        return false;
    }

    void Print(bool negative, IOutputStream& os) const override {
        os << (negative ? TStringBuf{"!~ "} : TStringBuf{"=~ "});
        os << '\'' << GetString() << '\'';
    }

    static RE2::Options Options() {
        RE2::Options opts;
        opts.set_never_capture(true);
        opts.set_case_sensitive(true);
        opts.set_one_line(true);
        opts.set_log_errors(false);
        return opts;
    }

private:
    re2::RE2 Regexp_;
};

class TAnyMatcher final: public IMatcher {
public:
    EMatcherType Type() const noexcept override {
        return EMatcherType::ANY;
    }

    TStringBuf Pattern() const noexcept override {
        return ANY_PATTERN;
    }

    bool Match(TStringBuf value) const override {
        return !value.empty();
    }

    bool MatchesAbsent() const override {
        return false;
    }

    void Print(bool negative, IOutputStream& os) const override {
        os << (negative ? TStringBuf{"!= '*'"} : TStringBuf{"= '*'"});
    }
};

class TAbsentMatcher final: public IMatcher {
public:
    EMatcherType Type() const noexcept override {
        return EMatcherType::ABSENT;
    }

    TStringBuf Pattern() const noexcept override {
        return ABSENT_PATTERN;
    }

    bool Match(TStringBuf value) const override {
        return value.empty();
    }

    bool MatchesAbsent() const override {
        return true;
    }

    void Print(bool negative, IOutputStream& os) const override {
        os << (negative ? TStringBuf{"!= '-'"} : TStringBuf{"= '-'"});
    }
};

class TMultiMatcher final: public IMultiMatcher, public TMatcherWithString<TMultiMatcher> {
public:
    TMultiMatcher(TStringBuf pattern, TVector<IMatcherPtr> matchers) noexcept
        : TMatcherWithString<TMultiMatcher>(pattern)
        , Matchers_(std::move(matchers))
    {
    }

    TStringBuf Pattern() const noexcept override {
        return GetString();
    }

    bool Match(TStringBuf value) const noexcept override {
        for (const auto& matcher: Matchers_) {
            if (matcher->Match(value)) {
                return true;
            }
        }
        return false;
    }

    bool MatchesAbsent() const noexcept override {
        for (const auto& matcher: Matchers_) {
            if (matcher->MatchesAbsent()) {
                return true;
            }
        }
        return false;
    }

    size_t Size() const noexcept override {
        return Matchers_.size();
    }

    const IMatcher* Get(size_t idx) const noexcept override {
        return Matchers_[idx].Get();
    }

    void Print(bool negative, IOutputStream& os) const override {
        os << (negative ? TStringBuf{"!= "} : TStringBuf{"= "});
        os << '\'' << GetString() << '\'';
    }

private:
    TVector<IMatcherPtr> Matchers_;
};

IMatcherPtr BasicMatcher(TStringBuf value) {
    if (value == ANY_PATTERN) {
        return AnyMatcher();
    }
    if (value == ABSENT_PATTERN) {
        return AbsentMatcher();
    }
    if (NSolomon::IsGlob(value)) {
        return GlobMatcher(value);
    }
    return ExactMatcher(value);
}

} // namespace

IMatcher::~IMatcher() noexcept {
}

IMatcherPtr Matcher(TStringBuf pattern) {
    if (!pattern.Contains('|')) {
        return BasicMatcher(pattern);
    }

    TVector<IMatcherPtr> matchers;
    for (auto it: StringSplitter(pattern).Split('|').SkipEmpty()) {
        matchers.push_back(BasicMatcher(it.Token()));
    }

    if (matchers.size() == 1) {
        return matchers[0];
    }

    return new (pattern.size()) TMultiMatcher(pattern, std::move(matchers));
}

IMatcherPtr ExactMatcher(TStringBuf pattern) {
    return new (pattern.size()) TExactMatcher(pattern);
}

IMatcherPtr GlobMatcher(TStringBuf pattern) {
    return new (pattern.size()) TGlobMatcher(pattern);
}

IMatcherPtr RegexMatcher(TStringBuf pattern) {
    return new (pattern.size()) TRegexMatcher(pattern);
}

IMatcherPtr AnyMatcher() {
    return new TAnyMatcher;
}

IMatcherPtr AbsentMatcher() {
    return new TAbsentMatcher;
}

} // namespace NSolomon
