#pragma once

#include <util/generic/strbuf.h>

namespace NSolomon {

/**
 * Type of matcher.
 */
enum class EMatcherType {
    EXACT,
    GLOB,
    REGEX,
    ANY,
    ABSENT,
    MULTI,
};

/**
 * Interface for particular matcher implementation.
 */
class IMatcher: public TAtomicRefCount<IMatcher> {
public:
    virtual ~IMatcher() noexcept;

    /**
     * @return type of current matcher.
     */
    virtual EMatcherType Type() const noexcept = 0;

    /**
     * @return string view of matching pattern.
     */
    virtual TStringBuf Pattern() const noexcept = 0;

    /**
     * Attempts to match the given string by underlying matcher.
     *
     * @param value  string to match.
     * @return {@code true } iff string matchers the matcher pattern.
     */
    virtual bool Match(TStringBuf value) const = 0;

    /**
     * Checks if the null value (the label is absent) satisfies the matcher
     *
     * @return {@code true } iff null value satisfies matchers the matcher pattern.
     */
    virtual bool MatchesAbsent() const = 0;

    /**
     * Outputs matcher as text into the given stream.
     *
     * @param os  output stream to print selector to.
     */
    virtual void Print(bool negative, IOutputStream& os) const = 0;
};

/**
 * Special kind of matcher that combines several basic matchers with logical OR operator,
 * so this matcher will match given string only if any of underlying basic matcher will
 * match it.
 *
 * Also this interface extends base interface to make it possible to access any underlying
 * basic matcher.
 */
class IMultiMatcher: public IMatcher {
public:
    /**
     * Final implementation that always returns MULTI type.
     */
    EMatcherType Type() const noexcept override final {
        return EMatcherType::MULTI;
    }

    /**
     * @return number of underlying basic matchers.
     */
    virtual size_t Size() const noexcept = 0;

    /**
     * Retrieves particular basic matcher.
     * @param idx  index of basic matcher.
     * @return not null pointer to basic matcher.
     */
    virtual const IMatcher* Get(size_t idx) const noexcept = 0;
};

using IMatcherPtr = TIntrusivePtr<IMatcher>;

/**
 * Choose the best matcher implementation by content of the given pattern.
 *
 * @param pattern   matcher pattern string
 * @return matcher object
 */
IMatcherPtr Matcher(TStringBuf pattern);

/**
 * Creates matcher which will checks string by complete match with pattern.
 * In most cases should not be used directly. Use {@link Matcher()) function instead.
 *
 * @param pattern   matcher pattern string
 * @return matcher object
 */
IMatcherPtr ExactMatcher(TStringBuf pattern);

/**
 * Creates matcher which will checks string matching by glob pattern.
 * In most cases should not be used directly. Use {@link Matcher()) function instead.
 *
 * @param pattern   matcher pattern string
 * @return matcher object
 */
IMatcherPtr GlobMatcher(TStringBuf pattern);

/**
 * Creates matcher which will checks string matching regular expression pattern.
 * In most cases should not be used directly. Use {@link Matcher()) function instead.
 *
 * @param pattern   matcher pattern string
 * @return matcher object
 */
IMatcherPtr RegexMatcher(TStringBuf pattern);

/**
 * Creates matcher which will match any non-empty string.
 * In most cases should not be used directly. Use {@link Matcher()) function instead.
 *
 * @param pattern   matcher pattern string
 * @return matcher object
 */
IMatcherPtr AnyMatcher();

/**
 * Creates matcher which will only match an empty string.
 * In most cases should not be used directly. Use {@link Matcher()) function instead.
 *
 * @param pattern   matcher pattern string
 * @return matcher object
 */
IMatcherPtr AbsentMatcher();

} // namespace NSolomon
