#include "selectors.h"

#include <solomon/libs/cpp/glob/glob.h>

#include <library/cpp/regex/pire/pcre2pire.h>

#include <util/generic/yexception.h>
#include <util/string/ascii.h>

#define SOLOMON_ENSURE_FTM(condition, message) \
    Y_ENSURE_EX((condition), ::NSolomon::TInvalidSelectorsFormat() << message)

namespace NSolomon {
namespace {

class TSelectorParser {
    enum class EOperator {
        EXACT,
        EQUAL,
        REGEX,
    };

    enum EPosition {
        Advance,
        Keep,
    };
public:
    explicit TSelectorParser(TStringBuf str) noexcept
        : Str_(str)
        , Pos_(0)
    {
    }

    TSelector ParseSelector() {
        TStringBuf key = NextToken();
        ConsumeSpaces();

        auto [negative, op] = ParseOperator();
        ConsumeSpaces();

        TStringBuf pattern = NextToken();
        ConsumeSpaces();

        return {key, CreateMatcher(op, pattern), negative};
    }

    TSelectors ParseSelectors() {
        TSelectors selectors;

        ConsumeSpaces();
        if (AtEnd()) {
            return selectors;
        }

        Consume('{');
        ConsumeSpaces();

        if (ReadNextCh<Keep>() != '}') {
            while (true) {
                selectors.Add(ParseSelector());

                if (AtEnd() || Str_[Pos_] == '}') {
                    break;
                }

                Consume(',');
                ConsumeSpaces();
            }
        }

        Consume('}');
        ConsumeSpaces();

        Y_ENSURE(AtEnd(), "Unexpected trailing chars at position " << Pos_ << ": " << Str_);
        return selectors;
    }

    void ConsumeSpaces() noexcept {
        while (!AtEnd() && IsAsciiSpace(Str_[Pos_])) {
            ++Pos_;
        }
    }

    bool AtEnd() const noexcept {
        return Pos_ >= Str_.size();
    }

private:
    static IMatcherPtr CreateMatcher(EOperator op, TStringBuf pattern) {
        switch (op) {
            case EOperator::EXACT:
                return ExactMatcher(pattern);

            case EOperator::REGEX: {
                TString pirePattern = Pcre2Pire(TString{pattern});
                if (auto glob = GlobFromRegex(pirePattern)) {
                    return Matcher(*glob);
                }
                return RegexMatcher(pirePattern);
            }

            case EOperator::EQUAL:
                return Matcher(pattern);
        }
    }

    TStringBuf NextToken() {
        char q = ReadNextCh<Keep>();
        if (q == '\'' || q == '\"') {
            // NOTE: escaping is not supported
            ++Pos_; // skip open quotes
            TStringBuf token = ReadUntil([q](char ch) { return ch == q; });
            ++Pos_; // skip close quotes
            return token;
        }

        return ReadUntil([](char ch) {
            // TODO: can be optimized using table lookup or bloom filter
            return IsAsciiSpace(ch) || ch == '=' || ch == '!' || ch == '~' || ch == ',' || ch == '{' || ch == '}';
        });
    }

    // supported operators: =, ==, =~, !=, !==, !~
    std::pair<bool, EOperator> ParseOperator() {
        enum class EPrefix {
            N,     // !
            NE,    // !=
            E,     // =
        };

        char ch = ReadNextCh<Advance>();
        EPrefix prefix =
                (ch == '!') ? EPrefix::N :
                (ch == '=') ? EPrefix::E : UnexpectedChar<EPrefix>(ch, "!=");

        while (Pos_ < Str_.size()) {
            ch = Str_[Pos_];
            switch (prefix) {
            case EPrefix::N:
                if (ch == '=') {
                    ++Pos_;
                    prefix = EPrefix::NE;
                } else if (ch == '~') {
                    ++Pos_;
                    return {true, EOperator::REGEX};
                } else {
                    UnexpectedChar(ch, "=~");
                }
                break;

            case EPrefix::NE:
                if (ch == '=') {
                    ++Pos_;
                    return {true, EOperator::EXACT};
                } else {
                    return {true, EOperator::EQUAL};
                }

            case EPrefix::E:
                if (ch == '=') {
                    ++Pos_;
                    return {false, EOperator::EXACT};
                } else if (ch == '~') {
                    ++Pos_;
                    return {false, EOperator::REGEX};
                } else {
                    return {false, EOperator::EQUAL};
                }
            }
        }

        ythrow TInvalidSelectorsFormat() << "Unable to parse operator at position " << Pos_ << ": " << Str_;
    }

    void Consume(char expectedCh) {
        char ch = ReadNextCh<Advance>();
        if (Y_UNLIKELY(ch != expectedCh)) {
            char expectedStr[2] = {expectedCh, '\0'};
            UnexpectedChar(ch, expectedStr);
        }
    }

    template <EPosition Position>
    char ReadNextCh() {
        SOLOMON_ENSURE_FTM(!AtEnd(), "Unexpected end of string at position " << Pos_ << ": " << Str_);

        char ch = Str_[Pos_];
        if constexpr (Position == Advance) {
            ++Pos_;
        }
        return ch;
    }

    template <typename TPredicate>
    TStringBuf ReadUntil(TPredicate&& p) {
        size_t begin = Pos_;
        while (!AtEnd() && !p(Str_[Pos_])) {
            ++Pos_;
        }
        return Str_.SubStr(begin, Pos_ - begin);
    }

    template <typename T = void>
    [[noreturn]] T UnexpectedChar(char ch, const char* expected) const {
        ythrow TInvalidSelectorsFormat()
                << "Unexpected char \'" << ch << "\' at position " << Pos_
                << " while expecting one of [" << expected << "]: " << Str_;
    }

private:
    const TStringBuf Str_;
    size_t Pos_;
};

} // namespace

IOutputStream& operator<<(IOutputStream& os, const TSelector& selector) {
    os << '\'' << selector.Key() << TStringBuf{"' "};
    selector.MatcherPtr()->Print(selector.Negative(), os);
    return os;
}

IOutputStream& operator<<(IOutputStream& os, const TSelectors& selectors) {
    os << '{';
    size_t i = 0;
    for (const TSelector& s: selectors) {
        if (i++ > 0) {
            os << TStringBuf{", "};
        }
        os << s;
    }
    return os << '}';
}

TSelector ParseSelector(TStringBuf str) {
    TSelectorParser parser{str};
    parser.ConsumeSpaces();
    SOLOMON_ENSURE_FTM(!parser.AtEnd(), "unexpected end of string while parsing single selector");
    auto selector = parser.ParseSelector();
    parser.ConsumeSpaces();
    SOLOMON_ENSURE_FTM(parser.AtEnd(), "unexpected chars at the end of string: " << str);
    return selector;
}

TSelectors ParseSelectors(TStringBuf str) {
    return TSelectorParser{str}.ParseSelectors();
}

} // namespace NSolomon
