#pragma once

#include "matchers.h"

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>

namespace NSolomon {

/**
 * TInvalidSelectorsFormat will be thrown when it is not possible to parse
 * selectors from string.
 */
class TInvalidSelectorsFormat: public yexception {
};

/**
 * Selector is a query part combining a key and a matcher.
 */
class TSelector {
public:
    TSelector() = default;

    TSelector(TStringBuf key, IMatcherPtr matcher, bool negative)
        : Key_(key)
        , Matcher_(std::move(matcher))
        , Negative_(negative)
    {
    }

    TSelector(TStringBuf key, TStringBuf value, bool negative)
        : TSelector(key, NSolomon::Matcher(value), negative)
    {
    }

    bool IsExact() const noexcept {
        return !Negative_ && Matcher_->Type() == EMatcherType::EXACT;
    }

    TStringBuf Key() const noexcept {
        return Key_;
    }

    TStringBuf Pattern() const noexcept {
        return Matcher_->Pattern();
    }

    EMatcherType Type() const noexcept {
        return Matcher_->Type();
    }

    bool Match(TStringBuf value) const {
        return Negative_ ^ Matcher_->Match(value);
    }

    bool Match(std::nullptr_t) {
        return Negative_ ^ Matcher_->Match(nullptr);
    }

    IMatcherPtr Matcher() const noexcept {
        return Matcher_;
    }

    const IMatcher* MatcherPtr() const noexcept {
        return Matcher_.Get();
    }

    const IMultiMatcher* MultiMatcherPtr() const noexcept {
        if (Matcher_->Type() != EMatcherType::MULTI)
            return nullptr;
        return static_cast<const IMultiMatcher*>(MatcherPtr());
    }

    bool Negative() const noexcept {
        return Negative_;
    }

private:
    TString Key_;
    IMatcherPtr Matcher_;
    bool Negative_;
};

/**
 * Associative-like container holding selectors for each key.
 */
class TSelectors: protected TVector<TSelector> {
    using TBase = TVector<TSelector>;

public:
    using iterator = TBase::iterator;
    using const_iterator = TBase::const_iterator;

public:
    bool Has(TStringBuf key) const noexcept {
        return Find(key) != cend();
    }

    bool Add(TStringBuf key, TStringBuf value, bool negative = false) {
        TBase::emplace_back(key, value, negative);
        return true;
    }

    bool Add(TSelector selector) {
        TBase::emplace_back(std::move(selector));
        return true;
    }

    void Override(TStringBuf key, TStringBuf value, bool negative = false) {
        Override(std::move(TSelector{key, value, negative}));
    }

    void Override(TSelector selector) {
        TStringBuf key = selector.Key();
        auto it = Find(key);
        if (it != end()) {
            *it = std::move(selector);
            Remove(std::next(it), end(), key);
        } else {
            TBase::emplace_back(std::move(selector));
        }
    }

    iterator Find(TBase::iterator beginIt, TBase::iterator endIt, TStringBuf key) noexcept {
        // NOTE: typical size of selectors is less than 16, so here used linear search,
        //       but further improvements can be achieved by using std::lower_bound and keeping
        //       selectors in sorted by keys order.
        return std::find_if(beginIt, endIt, [key](const auto& s) {
            return key == s.Key();
        });
    }

    const_iterator Find(TBase::const_iterator beginIt, TBase::const_iterator endIt, TStringBuf key) const noexcept {
        return std::find_if(beginIt, endIt, [key](const auto& s) {
            return key == s.Key();
        });
    }

    iterator Find(TStringBuf key) noexcept {
        return Find(begin(), end(), key);
    }

    const_iterator Find(TStringBuf key) const noexcept {
        return Find(cbegin(), cend(), key);
    }

    TBase::iterator Remove(TBase::iterator iter) {
        TBase::iterator lastIt = std::prev(end());
        bool moved = false;
        if (iter != lastIt) {
            *iter = std::move(*lastIt);
            moved = true;
        }
        pop_back();
        return moved ? iter : end();
    }

    bool Remove(TBase::iterator beginIt, TBase::iterator endIt, TStringBuf key) {
        bool result = false;
        while ((beginIt = Find(beginIt, endIt, key)) != endIt) {
            bool needUpdate = endIt == end();
            beginIt = Remove(beginIt);
            if (needUpdate) {
                endIt = end();
            }
            result = true;
        }
        return result;
    }

    bool Remove(TStringBuf key) {
        return Remove(begin(), end(), key);
    }

    // -- vector like operations --

    using TBase::front;
    using TBase::back;
    using TBase::operator[];
    using TBase::size;
    using TBase::empty;
    using TBase::clear;
    using TBase::begin;
    using TBase::end;
    using TBase::cbegin;
    using TBase::cend;
};

IOutputStream& operator<<(IOutputStream& os, const TSelector& selector);
IOutputStream& operator<<(IOutputStream& os, const TSelectors& selectors);

/**
 * Parses single selector from the given string and checks that there is only
 * one selector.
 *
 * @param str  text representation of selector
 * @return selectors object
 * @throws TInvalidSelectorsFormat
 */
TSelector ParseSelector(TStringBuf str);

/**
 * Parses all selectors from the given string.
 *
 * @param str  text representation of selectors
 * @return selectors object
 * @throws TInvalidSelectorsFormat
 */
TSelectors ParseSelectors(TStringBuf str);

} // namespace NSolomon
