#include <solomon/libs/cpp/selectors/matchers.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;

TString PrintToStr(const IMatcher& m, bool negative) {
    TStringStream ss;
    m.Print(negative, ss);
    return ss.Str();
}

TEST(TMatchersTest, Exact) {
    {
        auto m = ExactMatcher("");
        EXPECT_EQ(EMatcherType::EXACT, m->Type());
        EXPECT_EQ("", m->Pattern());
        EXPECT_TRUE(m->Match(""));
        EXPECT_FALSE(m->Match("a"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("== ''", PrintToStr(*m, false));
        EXPECT_EQ("!== ''", PrintToStr(*m, true));
    }
    {
        auto m = ExactMatcher("abc");
        EXPECT_EQ(EMatcherType::EXACT, m->Type());
        EXPECT_EQ("abc", m->Pattern());
        EXPECT_TRUE(m->Match("abc"));
        EXPECT_FALSE(m->Match("def"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("== 'abc'", PrintToStr(*m, false));
        EXPECT_EQ("!== 'abc'", PrintToStr(*m, true));
    }
}

TEST(TMatchersTest, Glob) {
    {
        auto m = GlobMatcher("");
        EXPECT_EQ(EMatcherType::GLOB, m->Type());
        EXPECT_EQ("", m->Pattern());
        EXPECT_TRUE(m->Match(""));
        EXPECT_FALSE(m->Match("a"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("= ''", PrintToStr(*m, false));
        EXPECT_EQ("!= ''", PrintToStr(*m, true));
    }
    {
        auto m = GlobMatcher("*");
        EXPECT_EQ(EMatcherType::GLOB, m->Type());
        EXPECT_EQ("*", m->Pattern());
        EXPECT_TRUE(m->Match("a"));
        EXPECT_TRUE(m->Match("b"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("= '*'", PrintToStr(*m, false));
        EXPECT_EQ("!= '*'", PrintToStr(*m, true));
    }
    {
        auto m = GlobMatcher("abc?ef");
        EXPECT_EQ(EMatcherType::GLOB, m->Type());
        EXPECT_EQ("abc?ef", m->Pattern());
        EXPECT_TRUE(m->Match("abcdef"));
        EXPECT_TRUE(m->Match("abcXef"));
        EXPECT_FALSE(m->Match("abcddddef"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("= 'abc?ef'", PrintToStr(*m, false));
        EXPECT_EQ("!= 'abc?ef'", PrintToStr(*m, true));
    }
    {
        auto m = GlobMatcher("abc*ef");
        EXPECT_EQ(EMatcherType::GLOB, m->Type());
        EXPECT_EQ("abc*ef", m->Pattern());
        EXPECT_TRUE(m->Match("abcddddef"));
        EXPECT_TRUE(m->Match("abcXXXXef"));
        EXPECT_FALSE(m->Match("abXXXXef"));
        EXPECT_EQ("= 'abc*ef'", PrintToStr(*m, false));
        EXPECT_EQ("!= 'abc*ef'", PrintToStr(*m, true));
    }
}

TEST(TMatchersTest, Regex) {
    {
        auto m = RegexMatcher("");
        EXPECT_EQ(EMatcherType::REGEX, m->Type());
        EXPECT_EQ("", m->Pattern());
        EXPECT_TRUE(m->Match(""));
        EXPECT_FALSE(m->Match("a"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("=~ ''", PrintToStr(*m, false));
        EXPECT_EQ("!~ ''", PrintToStr(*m, true));
    }
    {
        auto m = RegexMatcher("(a|b)c.ef{2}");
        EXPECT_EQ(EMatcherType::REGEX, m->Type());
        EXPECT_EQ("(a|b)c.ef{2}", m->Pattern());
        EXPECT_TRUE(m->Match("acXeff"));
        EXPECT_TRUE(m->Match("bcYeff"));
        EXPECT_FALSE(m->Match("abc"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("=~ '(a|b)c.ef{2}'", PrintToStr(*m, false));
        EXPECT_EQ("!~ '(a|b)c.ef{2}'", PrintToStr(*m, true));
    }
}

TEST(TMatchersTest, Any) {
    auto m = AnyMatcher();
    EXPECT_EQ(EMatcherType::ANY, m->Type());
    EXPECT_EQ("*", m->Pattern());
    EXPECT_FALSE(m->Match(""));
    EXPECT_TRUE(m->Match("a"));
    EXPECT_TRUE(m->Match("b"));
    EXPECT_FALSE(m->MatchesAbsent());
    EXPECT_EQ("= '*'", PrintToStr(*m, false));
    EXPECT_EQ("!= '*'", PrintToStr(*m, true));
}

TEST(TMatchersTest, Absent) {
    auto m = AbsentMatcher();
    EXPECT_EQ(EMatcherType::ABSENT, m->Type());
    EXPECT_EQ("-", m->Pattern());
    EXPECT_FALSE(m->Match("a"));
    EXPECT_FALSE(m->Match("b"));
    EXPECT_TRUE(m->Match(""));
    EXPECT_TRUE(m->MatchesAbsent());
    EXPECT_EQ("= '-'", PrintToStr(*m, false));
    EXPECT_EQ("!= '-'", PrintToStr(*m, true));
}

TEST(TMatchersTest, Matcher) {
    {
        auto m = Matcher("");
        EXPECT_EQ(EMatcherType::EXACT, m->Type());
        EXPECT_EQ("", m->Pattern());
    }
    {
        auto m = Matcher("abc");
        EXPECT_EQ(EMatcherType::EXACT, m->Type());
        EXPECT_EQ("abc", m->Pattern());
    }
    {
        auto m = Matcher("*");
        EXPECT_EQ(EMatcherType::ANY, m->Type());
        EXPECT_EQ("*", m->Pattern());
    }
    {
        auto m = Matcher("-");
        EXPECT_EQ(EMatcherType::ABSENT, m->Type());
        EXPECT_EQ("-", m->Pattern());
    }
    {
        auto m = Matcher("abc*ef");
        EXPECT_EQ(EMatcherType::GLOB, m->Type());
        EXPECT_EQ("abc*ef", m->Pattern());
    }
    {
        auto m = Matcher("a|b");
        EXPECT_EQ(EMatcherType::MULTI, m->Type());
        EXPECT_EQ("a|b", m->Pattern());
        EXPECT_TRUE(m->Match("a"));
        EXPECT_TRUE(m->Match("b"));
        EXPECT_FALSE(m->Match("c"));
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("= 'a|b'", PrintToStr(*m, false));
        EXPECT_EQ("!= 'a|b'", PrintToStr(*m, true));

        auto* mm = dynamic_cast<IMultiMatcher*>(m.Get());
        ASSERT_TRUE(mm);
        EXPECT_EQ(2u, mm->Size());
        EXPECT_EQ(EMatcherType::MULTI, mm->Type());

        auto* m1 = mm->Get(0);
        EXPECT_EQ(EMatcherType::EXACT, m1->Type());
        EXPECT_EQ("a", m1->Pattern());

        auto* m2 = mm->Get(1);
        EXPECT_EQ(EMatcherType::EXACT, m2->Type());
        EXPECT_EQ("b", m2->Pattern());
    }
    {
        auto m = Matcher("a|");
        EXPECT_EQ(EMatcherType::EXACT, m->Type());
        EXPECT_EQ("a", m->Pattern());
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("== 'a'", PrintToStr(*m, false));
        EXPECT_EQ("!== 'a'", PrintToStr(*m, true));
    }
    {
        auto m = Matcher("|b");
        EXPECT_EQ(EMatcherType::EXACT, m->Type());
        EXPECT_EQ("b", m->Pattern());
        EXPECT_FALSE(m->MatchesAbsent());
        EXPECT_EQ("== 'b'", PrintToStr(*m, false));
        EXPECT_EQ("!== 'b'", PrintToStr(*m, true));
    }
    {
        auto m = Matcher("-|b");
        EXPECT_EQ(EMatcherType::MULTI, m->Type());
        EXPECT_EQ("-|b", m->Pattern());
        EXPECT_FALSE(m->Match("a"));
        EXPECT_TRUE(m->Match("b"));
        EXPECT_TRUE(m->Match(""));
        EXPECT_TRUE(m->MatchesAbsent());
        EXPECT_EQ("= '-|b'", PrintToStr(*m, false));
        EXPECT_EQ("!= '-|b'", PrintToStr(*m, true));
    }
}
